<?php

namespace App\Http\Controllers;

use App\Bank;
use App\BankAccount;
use App\DepositRange;
use App\PaymentMethod;
use App\Player;
use App\Site;
use App\Transaction;
use App\WhiteListIp;
use Carbon\Carbon;
use Faker\Factory as Faker;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Redirector;
use Illuminate\Support\Str;
use Illuminate\View\View;

class PayControllerBackUp extends Controller
{
    public $supportLang = [
        'tr',
        'en'
    ];

    public $supportCurrency = [
        'TRY',
        'USD',
        'EUR'
    ];

    /**
     * @param $apiKey
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getDirectPage($apiKey, Request $request)
    {
        if (empty($apiKey))
            return view(env('THEME') . '.pay_error');

        $getSite = Site::where('code', $apiKey)->first();

        if (empty($getSite))
            return view(env('THEME') . '.pay_error');

        $faker = Faker::create();

        $sendData = [
            'api_id' => $getSite->id,
            'api_key' => $getSite->code,
            'user_id' => rand(100000, 9999999999),
            'external_transaction_id' => $faker->uuid,
            'username' => $faker->userName,
            'first_name' => $faker->firstName,
            'last_name' => $faker->lastName,
            'lang' => 'tr',
            'currency' => 'TRY',
            'return_url_success' => $getSite->return_url,
            'return_url_fail' => $getSite->return_url_fail
        ];
        $post = SeamlessController::postData(url('/payment/deposit/bank-transfer', [], true), $sendData);

        if (!empty($post['response']) && $post['response']->code == 200) {
            return redirect($post['response']->url);
        } else
            return view(env('THEME') . '.pay_error');
    }

    /**
     * @param $methodId
     * @param Request $request
     * @return Application|Factory|View
     */
    public function getPayPage($methodId, Request $request)
    {
        $hash = $request->input('hash');

        if (empty($hash))
            return view(env('THEME') . '.pay_error');

        $getTransaction = Transaction::where('hash', $hash)->first();

        if (empty($getTransaction))
            return view(env('THEME') . '.pay_error');

        if ($getTransaction->transaction_type != $methodId)
            Transaction::where('id', $getTransaction->id)->update([
                'transaction_type' => $methodId,
                'updated_at' => Carbon::now()
            ]);

        $getPaymentMethod = PaymentMethod::where('status', 1)
            ->where('id', $methodId)
            ->first();

        if (empty($getPaymentMethod))
            return view(env('THEME') . '.pay_error');

        if ($methodId == 1)
            $methodPage = 'wire_transfer';
        else if ($methodId == 2)
            $methodPage = 'mobile_bank';
        else if ($methodId == 3)
            $methodPage = 'papara';
        else
            $methodPage = null;

        if (empty($methodPage))
            return view(env('THEME') . '.pay_error');

        $getBanks = Bank::where('status', 1)->get();

        return view(env('THEME') . '.' . $methodPage)
            ->with('transaction', $getTransaction)
            ->with('paymentMethod', $getPaymentMethod)
            ->with('payHash', $hash)
            ->with('banks', $getBanks);
    }

    /**
     * @param Request $request
     * @return Application|Factory|View
     */
    public function getPayLobby(Request $request)
    {
        $hash = $request->input('hash');

        if (empty($hash))
            return view(env('THEME') . '.pay_error');

        $getTransaction = Transaction::where('hash', $hash)->first();

        if (empty($getTransaction))
            return view(env('THEME') . '.pay_error');

        $getPaymentMethods = PaymentMethod::where('status', 1)
            ->where('type', 1)
            ->orderBy('sort', 'ASC')
            ->get();

        if (empty($getPaymentMethods))
            return view(env('THEME') . '.pay_error');

        return view(env('THEME') . '.lobby')
            ->with('transaction', $getTransaction)
            ->with('paymentMethods', $getPaymentMethods)
            ->with('payHash', $hash);
    }

    /**
     * @return Application|Factory|View
     */
    public function getErrorPage()
    {
        return view(env('THEME') . '.pay_error');
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function getPayBankTransfer(Request $request)
    {
        $formData = $request->getContent();
        $decodeData = json_decode(base64_decode($formData), true);
        $explodeData = [];
        foreach ($decodeData as $k => $v) {
            $explodeData[$v['name']] = $v['value'];
        }

        if (empty($explodeData['site-bank']) || empty($explodeData['deposit-amount']))
            return response()->json([
                'valid' => 0,
                'validCode' => 10,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Tüm alanların doldurulması zorunludur!',
                'validClass' => 'error'
            ]);

        # Para Kontrolü
        if (!is_numeric($explodeData['deposit-amount']))
            return response()->json([
                'valid' => 0,
                'validCode' => 10,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Miktar sayısal bir değer içermelidir!',
                'validClass' => 'error'
            ]);

        $explodeData['deposit-amount'] = str_replace(',', '', $explodeData['deposit-amount']);
        $explodeData['deposit-amount'] = number_format($explodeData['deposit-amount'], 2, '.', '');

        # Tam Sayı Kontrolü
        if ((($explodeData['deposit-amount'] * 1000) % 100) !== 0)
            return response()->json([
                'valid' => 0,
                'validCode' => 10,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Miktar tam sayı olmalıdır!',
                'validClass' => 'error'
            ]);

        $getPaymentMethod = PaymentMethod::where('id', $explodeData['method-id'])->first();

        if (empty($getPaymentMethod))
            return response()->json([
                'valid' => 0,
                'validCode' => 10,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Ödeme yöntemi bulunamamıştır!',
                'validClass' => 'error'
            ]);

        if ($getPaymentMethod->status != 1)
            return response()->json([
                'valid' => 0,
                'validCode' => 10,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Ödeme yöntemi pasif edilmiştir!',
                'validClass' => 'error'
            ]);

        if ($explodeData['deposit-amount'] < $getPaymentMethod->min_amount)
            return response()->json([
                'valid' => 0,
                'validCode' => 10,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Minimum tutar yetersizdir!' . ' Minimum: ' . $getPaymentMethod->min_amount,
                'validClass' => 'error'
            ]);

        if ($explodeData['deposit-amount'] > $getPaymentMethod->max_amount)
            return response()->json([
                'valid' => 0,
                'validCode' => 10,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Maximum tutarı aştınız!' . ' Maximum: ' . $getPaymentMethod->max_amount,
                'validClass' => 'error'
            ]);

        if (env('BANK_TRANSFER_TYPE') == 1) {
            $updateTransaction = Transaction::where('hash', $explodeData['hash'])->update([
                'amount' => $explodeData['deposit-amount'],
                'bank_id' => $explodeData['site-bank'],
                'transaction_send_data' => json_encode($explodeData),
                'customer_note' => $explodeData['customer_note'],
                'player_approved' => 1
            ]);

            if ($updateTransaction)
                return response()->json([
                    'valid' => 1,
                    'validCode' => 200,
                    'validTitle' => 'Lütfen bekleyiniz...',
                    'validMessage' => 'Para yatırma işleminiz başlatılmıştır!',
                    'validClass' => 'info'
                ]);
            else
                return response()->json([
                    'valid' => 0,
                    'validCode' => 12,
                    'validTitle' => 'Hata var!',
                    'validMessage' => 'Yatırım işlemi oluşturulamadı!',
                    'validClass' => 'error'
                ]);
        } else if (env('BANK_TRANSFER_TYPE') == 2) {
            $getBankAccount = BankAccount::where('bank_id', $explodeData['site-bank'])
                ->where('type_id', 1)
                ->where('status', 1)
                ->first();

            if (empty($getBankAccount))
                return response()->json([
                    'valid' => 0,
                    'validCode' => 12,
                    'validTitle' => 'Hata var!',
                    'validMessage' => 'Aktif banka hesabı bulunamamıştır!',
                    'validClass' => 'error'
                ]);

            $updateTransaction = Transaction::where('hash', $explodeData['hash'])->update([
                'share_bank_account_id' => $getBankAccount->id,
                'amount' => $explodeData['deposit-amount']
            ]);

            if ($updateTransaction)
                return response()->json([
                    'valid' => 1,
                    'validCode' => 200,
                    'validTitle' => 'Lütfen bekleyiniz...',
                    'validMessage' => 'Para yatırma işleminiz başlatılmıştır!',
                    'validClass' => 'info'
                ]);
            else
                return response()->json([
                    'valid' => 0,
                    'validCode' => 12,
                    'validTitle' => 'Hata var!',
                    'validMessage' => 'Yatırım işlemi oluşturulamadı!',
                    'validClass' => 'error'
                ]);
        } else {
            return response()->json([
                'valid' => 0,
                'validCode' => 12,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Bilinmeyen transfer yöntemi kullanılmıştır!',
                'validClass' => 'error'
            ]);
        }
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function getCheckBankTransfer(Request $request)
    {
        #$json = json_decode('{"valid" : 1 , "validCode" : 205 , "validTitle" : "Lütfen bekleyiniz..." , "validMessage" : "İşleminiz için uygun banka hesap bilgisi hazırlanıyor... Lütfen sayfadan ayrılmayınız..." , "validClass" : "info" , "base64Data" : ""}');
        #$json = json_decode('{"valid" : 1 , "validCode" : 200 , "validTitle" : "Yatırım bilgileri" , "validMessage" : "Lütfen aşağıdaki Banka hesabına EFT ile yatırım gerçekleştiriniz..." , "validClass" : "success" , "base64Data" : "PHA+TMO8dGZlbiBhxZ9hxJ/EsWRha2kgQmFua2EgaGVzYWLEsW5hIDx1PkVGVDwvdT4gaWxlIDxzdHJvbmc+Mi41MDAsMDAgVEw8L3N0cm9uZz4geWF0xLFyxLFtIGdlcsOnZWtsZcWfdGlyaW5pei4uLjwvcD48cD48c3Ryb25nPkVGVCB0cmFuc2ZlciBhw6fEsWtsYW1hbGFyxLFuZGEgc2FkZWNlIGtlbmRpIGlzbWluaXogc295aXNtaW5peiB5YXrEsWxtYWzEsWTEsXIgdmV5YSBib8WfIGLEsXJha8SxbG1hbMSxZMSxciBha3NpIHRha2RpcmRlIHlhdMSxcsSxbcSxbsSxeiBrYWJ1bCBlZGlsbWV5ZWNla3Rpci48L3N0cm9uZz48L3A+PHA+QmFua2E6IDxzdHJvbmc+WmlyYWF0IEJhbmthc8SxPC9zdHJvbmc+PC9wPjxwPkhlc2FwIHNhaGliaTogPHN0cm9uZz5PS0FOIMWeRU5Hw5xSPC9zdHJvbmc+PC9wPjxwPsWedWJlIEtvZHU6IDxzdHJvbmc+MTY5NDwvc3Ryb25nPjwvcD48cD5IZXNhcCBObzogPHN0cm9uZz44OTY3MzcyNDwvc3Ryb25nPjwvcD48cD5JQkFOOiA8c3Ryb25nPlRSIDY4MDAgMDEwMCAxNjk0IDg5NjcgMzcyNCA1MDAxPC9zdHJvbmc+PC9wPjxocj48cD5Mw7x0ZmVuIHlhdMSxcsSxbSB5YXBtYWRhbiA8c3Ryb25nPllhdMSxcsSxbcSxIGdlcsOnZWtsZcWfdGlyZGltPC9zdHJvbmc+IGJ1dG9udW5hIHTEsWtsYW1hecSxbsSxei4uLjwvcD48YSBvbmNsaWNrPSJCYW5rVHJhbnNmZXJDaGVja0RpZCgpIiBocmVmPSJqYXZhc2NyaXB0OjsiIGNsYXNzPSJidG4gYnRuLXN1Y2Nlc3MiIHN0eWxlPSJjb2xvcjp3aGl0ZTsiPllhdMSxcsSxbcSxIGdlcsOnZWtsZcWfdGlyZGltITwvYT4="}');

        $formData = $request->getContent();
        $decodeData = json_decode(base64_decode($formData), true);

        if (empty($decodeData))
            return response()->json([
                'valid' => 0,
                'validCode' => 10,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Tüm alanların doldurulması zorunludur!',
                'validClass' => 'error'
            ]);

        $explodeData = [];
        foreach ($decodeData as $k => $v) {
            $explodeData[$v['name']] = $v['value'];
        }

        $getTransaction = Transaction::where('hash', $explodeData['hash'])->first();

        if ($getTransaction->player_approved == 2)
            return response()->json([
                'valid' => 1,
                'validCode' => 755,
                'validTitle' => 'İşleminiz İptal Edildi',
                'validMessage' => null,
                'validClass' => 'error'
            ]);

        if ($getTransaction->is_success == 1)
            return response()->json([
                'valid' => 1,
                'validCode' => 253,
                'validTitle' => 'Yatırımınız Onaylanmıştır.',
                'validMessage' => 'Yatırım miktarı hesabınıza aktarılmıştır, siteye geri dönmek istiyor musunuz ?',
                'validClass' => 'success'
            ]);

        if ($getTransaction->is_success == 2)
            return response()->json([
                'valid' => 1,
                'validCode' => 252,
                'validTitle' => 'Yatırımınız Bulunamadı.',
                'validMessage' => 'Yatırım yok, siteye geri dönmek istiyor musunuz ?',
                'validClass' => 'error'
            ]);

        if ($getTransaction->player_approved == 1)
            return response()->json([
                'valid' => 1,
                'validCode' => 290,
                'validTitle' => 'Kontrol ediliyor',
                'validMessage' => 'Yatırımınız kontrol ediliyor lütfen bekleyiniz...',
                'validClass' => 'info'
            ]);

        if (empty($getTransaction->share_bank_account_id))
            return response()->json([
                'valid' => 1,
                'validCode' => 205,
                'validTitle' => 'Lütfen bekleyiniz...',
                'validMessage' => 'İşleminiz için uygun banka hesap bilgisi hazırlanıyor... Lütfen sayfadan ayrılmayınız...',
                'validClass' => 'info',
                'base64Data' => ''
            ]);

        // Payment Check
        if ($request->input('action') == 'did') {
            $updateTransaction = Transaction::where('id', $getTransaction->id)->update([
                'player_approved' => 1
            ]);

            return response()->json([
                'valid' => 1,
                'validCode' => 290,
                'validTitle' => 'Kontrol ediliyor',
                'validMessage' => 'Yatırımınız kontrol ediliyor lütfen bekleyiniz...',
                'validClass' => 'info'
            ]);
        }

        $getBankAccount = BankAccount::where('id', $getTransaction->share_bank_account_id)
            ->where('status', 1)
            ->first();

        if (empty($getBankAccount))
            return response()->json([
                'valid' => 1,
                'validCode' => 205,
                'validTitle' => 'Lütfen bekleyiniz...',
                'validMessage' => 'İşleminiz için uygun banka hesap bilgisi hazırlanıyor... Lütfen sayfadan ayrılmayınız...',
                'validClass' => 'info',
                'base64Data' => ''
            ]);

        $htmlFormat = '
            <p>Lütfen aşağıdaki Banka hesabına <u>Havale/EFT</u> ile <strong>' . $getTransaction->amount . ' TL</strong> yatırım gerçekleştiriniz...</p>
            <p><strong>Havale/EFT transfer açıklamalarında sadece kendi isminiz soyisminiz yazılmalıdır veya boş bırakılmalıdır aksi takdirde yatırımınız kabul edilmeyecektir.</strong></p>
            <p>Banka: <strong>' . $getBankAccount->banks->name . '</strong></p>
            <p>Hesap sahibi: <strong>' . $getBankAccount->account_owner . '</strong></p>
            <p>Şube Kodu: <strong>' . $getBankAccount->branch_code . '</strong></p>
            <p>Hesap No: <strong>' . $getBankAccount->account_number . '</strong></p>
            <p>IBAN: <strong>' . $getBankAccount->iban . '</strong></p>
            <hr>
            <p>Lütfen yatırım yapmadan <strong>Yatırımı gerçekleştirdim</strong> butonuna tıklamayınız...</p>
            <a onclick="BankTransferCheckDid()" href="javascript:;" class="btn btn-success" style="color:white;">Yatırımı gerçekleştirdim!</a>
        ';

        $json = [
            'valid' => 1,
            'validCode' => 200,
            'validTitle' => 'Yatırım bilgileri',
            'validMessage' => 'Lütfen aşağıdaki Banka hesabına Havale/EFT ile yatırım gerçekleştiriniz...',
            'validClass' => 'success',
            'base64Data' => base64_encode($htmlFormat)
        ];

        return response()->json($json);
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function getPayPaParaTransfer(Request $request)
    {
        $formData = $request->getContent();
        $decodeData = json_decode(base64_decode($formData), true);
        $explodeData = [];
        foreach ($decodeData as $k => $v) {
            $explodeData[$v['name']] = $v['value'];
        }

        if (empty($explodeData['site-fullname']) || empty($explodeData['deposit-amount']))
            return response()->json([
                'valid' => 0,
                'validCode' => 10,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Tüm alanların doldurulması zorunludur!',
                'validClass' => 'error'
            ]);

        # Para Kontrolü
        if (!is_numeric($explodeData['deposit-amount']))
            return response()->json([
                'valid' => 0,
                'validCode' => 10,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Miktar sayısal bir değer içermelidir!',
                'validClass' => 'error'
            ]);

        $explodeData['deposit-amount'] = str_replace(',', '', $explodeData['deposit-amount']);
        $explodeData['deposit-amount'] = number_format($explodeData['deposit-amount'], 2, '.', '');

        # Tam Sayı Kontrolü
        if ((($explodeData['deposit-amount'] * 1000) % 100) !== 0)
            return response()->json([
                'valid' => 0,
                'validCode' => 10,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Miktar tam sayı olmalıdır!',
                'validClass' => 'error'
            ]);

        $getTransaction = Transaction::where('hash', $explodeData['hash'])->first();

        if (empty($getTransaction))
            return response()->json([
                'valid' => 0,
                'validCode' => 10,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Ödeme sayfası bulunamamıştır!',
                'validClass' => 'error'
            ]);

        /*if (
            $getTransaction->player->first_name . ' ' . $getTransaction->player->last_name != $explodeData['site-fullname']
        ) {
            $nameService = $this->nameChecker($explodeData['site-fullname']);

            Player::where('id', $getTransaction->player_id)->update([
                'first_name' => $nameService['first_name'],
                'last_name' => $nameService['last_name'],
                'updated_at' => Carbon::now()
            ]);
        }*/

        if (
            $getTransaction->player->first_name . ' ' . $getTransaction->player->last_name != $explodeData['site-fullname'] &&
            $getTransaction->api_id == 3
        ) {
            $nameService = $this->nameChecker($explodeData['site-fullname']);

            Player::where('id', $getTransaction->player_id)->update([
                'first_name' => $nameService['first_name'],
                'last_name' => $nameService['last_name'],
                'updated_at' => Carbon::now()
            ]);
        }

        $getPaymentMethod = PaymentMethod::where('id', $explodeData['method-id'])->first();

        if (empty($getPaymentMethod))
            return response()->json([
                'valid' => 0,
                'validCode' => 10,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Ödeme yöntemi bulunamamıştır!',
                'validClass' => 'error'
            ]);

        if ($getPaymentMethod->status != 1)
            return response()->json([
                'valid' => 0,
                'validCode' => 10,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Ödeme yöntemi pasif edilmiştir!',
                'validClass' => 'error'
            ]);

        if ($explodeData['deposit-amount'] < $getPaymentMethod->min_amount)
            return response()->json([
                'valid' => 0,
                'validCode' => 10,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Minimum tutar yetersizdir!' . ' Minimum: ' . $getPaymentMethod->min_amount,
                'validClass' => 'error'
            ]);

        if ($explodeData['deposit-amount'] > $getPaymentMethod->max_amount)
            return response()->json([
                'valid' => 0,
                'validCode' => 10,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Maximum tutarı aştınız!' . ' Maximum: ' . $getPaymentMethod->max_amount,
                'validClass' => 'error'
            ]);

        if (env('PAPARA_TYPE') == 1) {
            $updateTransaction = Transaction::where('hash', $explodeData['hash'])->update([
                'amount' => $explodeData['deposit-amount'],
                'transaction_send_data' => json_encode($explodeData),
                'customer_note' => $explodeData['customer_note'],
                'player_approved' => 1
            ]);

            if ($updateTransaction)
                return response()->json([
                    'valid' => 1,
                    'validCode' => 200,
                    'validTitle' => 'Lütfen bekleyiniz...',
                    'validMessage' => 'Para yatırma işleminiz başlatılmıştır!',
                    'validClass' => 'info'
                ]);
            else
                return response()->json([
                    'valid' => 0,
                    'validCode' => 12,
                    'validTitle' => 'Hata var!',
                    'validMessage' => 'Yatırım işlemi oluşturulamadı!',
                    'validClass' => 'error'
                ]);
        } else if (env('PAPARA_TYPE') == 2 || env('PAPARA_TYPE') == 3) {
            $getApiId = (int)$getTransaction->api_id;
            $getActiveBankAccounts = BankAccount::where('type_id', 2)
                ->where('status', 1)
                //->whereIn('api_id', [$getApiId])
                ->where('api_id', 'LIKE', "%$getApiId,%")
                ->select(
                    'id',
                    'api_id',
                    'process_limit',
                    'total_limit',
                    'daily_limit',
                    'deposit_range_id',
                    'mail_server'
                )
                ->get();

            if ($getActiveBankAccounts->isEmpty()) {
                $getActiveBankAccounts = BankAccount::where('type_id', 2)
                    ->where('status', 1)
                    ->select(
                        'id',
                        'api_id',
                        'process_limit',
                        'total_limit',
                        'daily_limit',
                        'deposit_range_id',
                        'mail_server'
                    )
                    ->get();
            }

            if (empty($getActiveBankAccounts))
                return response()->json([
                    'valid' => 0,
                    'validCode' => 12,
                    'validTitle' => 'Hata var!',
                    'validMessage' => 'Aktif papara hesabı bulunamamıştır!',
                    'validClass' => 'error'
                ]);

            $activeBankId = 0;
            $dataBankAccounts = [];
            foreach ($getActiveBankAccounts as $bankAccount) {
                // Eğer ki otomatik papara ise mail server seçimi boş ise atla
                if (env('PAPARA_TYPE') == 3 && empty($bankAccount->mail_server))
                    continue;

                // Yatırım aralığı kontrol ediliyor
                if (!empty($bankAccount->deposit_range_id) && $bankAccount->deposit_range_id > 0) {
                    $getDepositRange = DepositRange::findDepositRangeByAmount($explodeData['deposit-amount']);

                    if (!empty($getDepositRange)) {
                        if ($bankAccount->deposit_range_id == $getDepositRange->id) {
                            $dataBankAccounts[] = $this->getCheckLimit($bankAccount);
                        }
                    }
                } else {
                    $dataBankAccounts[] = $this->getCheckLimit($bankAccount);
                }
            }

            if (!empty($dataBankAccounts) && $dataBankAccounts > 0) {
                shuffle($dataBankAccounts);
                $activeBankId = $dataBankAccounts[0];
            }

            if (empty($activeBankId) || $activeBankId == 0)
                return response()->json([
                    'valid' => 0,
                    'validCode' => 12,
                    'validTitle' => 'Hata var!',
                    'validMessage' => 'Aktif papara hesabı bulunamamıştır!',
                    'validClass' => 'error'
                ]);

            $getBankAccount = BankAccount::where('id', $activeBankId)->where('status', 1)->first();

            if (empty($getBankAccount))
                return response()->json([
                    'valid' => 0,
                    'validCode' => 12,
                    'validTitle' => 'Hata var!',
                    'validMessage' => 'Aktif papara hesabı bulunamamıştır!',
                    'validClass' => 'error'
                ]);

            $updateTransaction = Transaction::updateTransactionByHash($explodeData['hash'], [
                'share_bank_account_id' => $getBankAccount->id,
                'amount' => $explodeData['deposit-amount']
            ]);

            if ($updateTransaction)
                return response()->json([
                    'valid' => 1,
                    'validCode' => 200,
                    'validTitle' => 'Lütfen bekleyiniz...',
                    'validMessage' => 'Para yatırma işleminiz başlatılmıştır!',
                    'validClass' => 'info'
                ]);
            else
                return response()->json([
                    'valid' => 0,
                    'validCode' => 12,
                    'validTitle' => 'Hata var!',
                    'validMessage' => 'Yatırım işlemi oluşturulamadı!',
                    'validClass' => 'error'
                ]);
        } else {
            return response()->json([
                'valid' => 0,
                'validCode' => 12,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Bilinmeyen transfer yöntemi kullanılmıştır!',
                'validClass' => 'error'
            ]);
        }
    }

    /**
     * @param Request $request
     * @return JsonResponse|void
     */
    public function getCheckPaParaTransfer(Request $request)
    {
        #$json = json_decode('{"valid" : 1 , "validCode" : 205 , "validTitle" : "Lütfen bekleyiniz..." , "validMessage" : "İşleminiz için uygun banka hesap bilgisi hazırlanıyor... Lütfen sayfadan ayrılmayınız..." , "validClass" : "info" , "base64Data" : ""}');
        #$json = json_decode('{"valid" : 1 , "validCode" : 200 , "validTitle" : "Yatırım bilgileri" , "validMessage" : "Lütfen aşağıdaki Banka hesabına EFT ile yatırım gerçekleştiriniz..." , "validClass" : "success" , "base64Data" : "PHA+TMO8dGZlbiBhxZ9hxJ/EsWRha2kgQmFua2EgaGVzYWLEsW5hIDx1PkVGVDwvdT4gaWxlIDxzdHJvbmc+Mi41MDAsMDAgVEw8L3N0cm9uZz4geWF0xLFyxLFtIGdlcsOnZWtsZcWfdGlyaW5pei4uLjwvcD48cD48c3Ryb25nPkVGVCB0cmFuc2ZlciBhw6fEsWtsYW1hbGFyxLFuZGEgc2FkZWNlIGtlbmRpIGlzbWluaXogc295aXNtaW5peiB5YXrEsWxtYWzEsWTEsXIgdmV5YSBib8WfIGLEsXJha8SxbG1hbMSxZMSxciBha3NpIHRha2RpcmRlIHlhdMSxcsSxbcSxbsSxeiBrYWJ1bCBlZGlsbWV5ZWNla3Rpci48L3N0cm9uZz48L3A+PHA+QmFua2E6IDxzdHJvbmc+WmlyYWF0IEJhbmthc8SxPC9zdHJvbmc+PC9wPjxwPkhlc2FwIHNhaGliaTogPHN0cm9uZz5PS0FOIMWeRU5Hw5xSPC9zdHJvbmc+PC9wPjxwPsWedWJlIEtvZHU6IDxzdHJvbmc+MTY5NDwvc3Ryb25nPjwvcD48cD5IZXNhcCBObzogPHN0cm9uZz44OTY3MzcyNDwvc3Ryb25nPjwvcD48cD5JQkFOOiA8c3Ryb25nPlRSIDY4MDAgMDEwMCAxNjk0IDg5NjcgMzcyNCA1MDAxPC9zdHJvbmc+PC9wPjxocj48cD5Mw7x0ZmVuIHlhdMSxcsSxbSB5YXBtYWRhbiA8c3Ryb25nPllhdMSxcsSxbcSxIGdlcsOnZWtsZcWfdGlyZGltPC9zdHJvbmc+IGJ1dG9udW5hIHTEsWtsYW1hecSxbsSxei4uLjwvcD48YSBvbmNsaWNrPSJCYW5rVHJhbnNmZXJDaGVja0RpZCgpIiBocmVmPSJqYXZhc2NyaXB0OjsiIGNsYXNzPSJidG4gYnRuLXN1Y2Nlc3MiIHN0eWxlPSJjb2xvcjp3aGl0ZTsiPllhdMSxcsSxbcSxIGdlcsOnZWtsZcWfdGlyZGltITwvYT4="}');

        $formData = $request->getContent();
        $decodeData = json_decode(base64_decode($formData), true);

        if (empty($decodeData))
            return response()->json([
                'valid' => 0,
                'validCode' => 10,
                'validTitle' => 'Hata var!',
                'validMessage' => 'Tüm alanların doldurulması zorunludur!',
                'validClass' => 'error'
            ]);

        $explodeData = [];
        foreach ($decodeData as $k => $v) {
            $explodeData[$v['name']] = $v['value'];
        }

        $getTransaction = Transaction::where('hash', $explodeData['hash'])->first();

        if ($getTransaction->player_approved == 2)
            return response()->json([
                'valid' => 1,
                'validCode' => 755,
                'validTitle' => 'İşleminiz İptal Edildi',
                'validMessage' => null,
                'validClass' => 'error'
            ]);

        if ($getTransaction->is_success == 1)
            return response()->json([
                'valid' => 1,
                'validCode' => 253,
                'validTitle' => 'Yatırımınız Onaylanmıştır.',
                'validMessage' => 'Yatırım miktarı hesabınıza aktarılmıştır, siteye geri dönmek istiyor musunuz ?',
                'validClass' => 'success'
            ]);

        if ($getTransaction->is_success == 2)
            return response()->json([
                'valid' => 1,
                'validCode' => 252,
                'validTitle' => 'Yatırımınız Bulunamadı.',
                'validMessage' => 'Yatırım yok, siteye geri dönmek istiyor musunuz ?',
                'validClass' => 'error'
            ]);

        if (env('PAPARA_TYPE') == 1) {
            if ($getTransaction->player_approved == 1)
                return response()->json([
                    'valid' => 1,
                    'validCode' => 211,
                    'validTitle' => 'Kontrol ediliyor',
                    'validMessage' => 'Yatırımınız kontrol ediliyor lütfen bekleyiniz...',
                    'validClass' => 'info'
                ]);

            if (empty($getTransaction->share_bank_account_id))
                return response()->json([
                    'valid' => 1,
                    'validCode' => 205,
                    'validTitle' => 'Lütfen bekleyiniz...',
                    'validMessage' => 'İşleminiz için uygun papara hesap bilgisi hazırlanıyor... Lütfen sayfadan ayrılmayınız...',
                    'validClass' => 'info',
                    'base64Data' => ''
                ]);

            // Payment Check
            if ($request->input('action') == 'online') {
                return response()->json([
                    'valid' => 1,
                    'validCode' => 200,
                    'validTitle' => 'OK',
                    'validMessage' => 'OK',
                    'validClass' => 'success'
                ]);
            }

            // Payment Check
            if ($request->input('action') == 'did') {
                $updateTransaction = Transaction::where('id', $getTransaction->id)->update([
                    'player_approved' => 1
                ]);

                return response()->json([
                    'valid' => 1,
                    'validCode' => 290,
                    'validTitle' => 'Kontrol ediliyor',
                    'validMessage' => 'Yatırımınız kontrol ediliyor lütfen bekleyiniz...',
                    'validClass' => 'info'
                ]);
            }

            // Payment Declined
            if ($request->input('action') == 'desist') {
                $updateTransaction = Transaction::where('id', $getTransaction->id)->update([
                    'is_success' => 2,
                    'player_approved' => 2
                ]);

                $parameters = [
                    'apiId' => $getTransaction->sites->id,
                    'apiKey' => $getTransaction->sites->code,
                    'paymentId' => $getTransaction->id,
                    'externalTransactionId' => $getTransaction->external_transaction_id,
                    'userId' => $getTransaction->player->external_user_id,
                    'username' => $getTransaction->player->username,
                    'type' => 'deposit',
                    'amount' => $getTransaction->amount,
                    'status' => false,
                    'message' => 'success',
                    'hash' => $getTransaction->hash
                ];
                $postData = SeamlessController::postData($getTransaction->sites->seamless_url, $parameters);

                return response()->json([
                    'valid' => 1,
                    'validCode' => 252,
                    'validTitle' => 'İşleminiz İptal Edildi',
                    'validMessage' => 'Siteye geri dönmek ister misiniz ?',
                    'validClass' => 'error'
                ]);
            }

            $getBankAccount = BankAccount::where('id', $getTransaction->share_bank_account_id)
                ->where('status', 1)
                ->first();

            if (empty($getBankAccount)) {
                /*$updateTransaction = Transaction::where('id', $getTransaction->id)->update([
                    'is_success' => 2,
                    'player_approved' => 2
                ]);

                return response()->json([
                    'valid' => 1,
                    'validCode' => 252,
                    'validTitle' => 'İşleminiz Zaman Aşımına Uğradığı İçin İptal Edildi',
                    'validMessage' => 'Siteye geri dönmek ister misiniz ?',
                    'validClass' => 'error'
                ]);*/

                return response()->json([
                    'valid' => 1,
                    'validCode' => 205,
                    'validTitle' => 'Lütfen bekleyiniz...',
                    'validMessage' => 'İşleminiz için uygun papara hesap bilgisi hazırlanıyor... Lütfen sayfadan ayrılmayınız...',
                    'validClass' => 'info',
                    'base64Data' => ''
                ]);
            }

            $htmlFormat = '
            <h6 class="alert alert-danger">İŞLEM SÜRESİNCE <u>KESİNLİKLE</u> SAYFADAN AYRILMAYINIZ.</h6>
            <p>Lütfen aşağıdaki Papara hesabına <strong>' . $getTransaction->amount . ' TL</strong> yatırım gerçekleştiriniz...</p>
            <p>Hesap sahibi: <strong>' . $getBankAccount->account_owner . '</strong></p>
            <p>Hesap No: <strong>' . $getBankAccount->account_number . '</strong></p>
            <hr>
            <p><b>QR Okutarak daha hızlı para transferi yapabilirsiniz</b></p>
            <p><img src="https://chart.googleapis.com/chart?chs=200x200&chld=M|0&cht=qr&chl=https://www.papara.com/personal/qr?accountNumber=' . $getBankAccount->account_number . '%26amount=' . $getTransaction->amount . '" /></p>
            <hr>
            <p style="color:red;">Lütfen yatırım yapmadan <strong>Yatırımı gerçekleştirdim</strong> butonuna tıklamayınız...</p>
            <div class="row">
                <div class="col-md-6 mb-1 mt-1">
                    <a onclick="PaparaCheckDesist()" href="javascript:;" class="btn btn-danger btn-block" style="color:white;">Vazgeç!</a>
                </div>
                <div class="col-md-6 mb-1 mt-1">
                    <a onclick="PaparaCheckDid()" href="javascript:;" class="btn btn-success btn-block" style="color:white;">Yatırımı gerçekleştirdim!</a>
                </div>
            </div>
        ';

            $json = [
                'valid' => 1,
                'validCode' => 200,
                'validTitle' => 'Yatırım bilgileri',
                'validMessage' => 'Lütfen aşağıdaki Banka hesabına Havale/EFT ile yatırım gerçekleştiriniz...',
                'validClass' => 'success',
                'base64Data' => base64_encode($htmlFormat)
            ];

            return response()->json($json);
        } else if (env('PAPARA_TYPE') == 2 || env('PAPARA_TYPE') == 3) {
            if ($getTransaction->player_approved == 1)
                return response()->json([
                    'valid' => 1,
                    'validCode' => 211,
                    'validTitle' => 'Kontrol ediliyor',
                    'validMessage' => 'Yatırımınız kontrol ediliyor lütfen bekleyiniz...',
                    'validClass' => 'info'
                ]);

            if (empty($getTransaction->share_bank_account_id))
                return response()->json([
                    'valid' => 1,
                    'validCode' => 205,
                    'validTitle' => 'Lütfen bekleyiniz...',
                    'validMessage' => 'İşleminiz için uygun papara hesap bilgisi hazırlanıyor... Lütfen sayfadan ayrılmayınız...',
                    'validClass' => 'info',
                    'base64Data' => ''
                ]);

            // Payment Check
            if ($request->input('action') == 'online') {
                Transaction::where('id', $getTransaction->id)->update([
                    'updated_at' => Carbon::now()
                ]);

                return response()->json([
                    'valid' => 1,
                    'validCode' => 200,
                    'validTitle' => 'OK',
                    'validMessage' => 'OK',
                    'validClass' => 'success'
                ]);
            }

            // Payment Check
            if ($request->input('action') == 'did') {
                if (env('PAPARA_TYPE') == 3) {
                    $siteFullName = $this->turkishStrToUpper($getTransaction->player->first_name . ' ' . $getTransaction->player->last_name);

                    if (empty($siteFullName))
                        return response()->json([
                            'valid' => 1,
                            'validCode' => 211,
                            'validTitle' => 'Kontrol ediliyor',
                            'validMessage' => 'Yatırımınız kontrol ediliyor lütfen bekleyiniz...',
                            'validClass' => 'info'
                        ]);

                    $getBankAccountDetails = BankAccount::where('id', $getTransaction->share_bank_account_id)->first();
                    $mailPath = BankAccount::getMailServer($getBankAccountDetails->mail_server);
                    $connect = PaParaController::connect(
                        $mailPath,
                        $getBankAccountDetails->mail_username,
                        $getBankAccountDetails->mail_password,
                        1,
                        20
                    );

                    if ($connect['status'] == true && !empty($connect['data'])) {
                        foreach ($connect['data'] as $key => $value) {
                            if ($value['status'] == true) {
                                $autoSender = $this->turkishStrToUpper($value['sender']);
                                $autoAmount = $value['money'];

                                # Find Transaction
                                if ($siteFullName == $autoSender && $getTransaction->amount == $autoAmount) {
                                    # Seamless (CallBack) Send Request
                                    $parameters = [
                                        'apiId' => $getTransaction->sites->id,
                                        'apiKey' => $getTransaction->sites->code,
                                        'paymentId' => $getTransaction->id,
                                        'externalTransactionId' => $getTransaction->external_transaction_id,
                                        'userId' => $getTransaction->player->external_user_id,
                                        'username' => $getTransaction->player->username,
                                        'type' => 'deposit',
                                        'amount' => $getTransaction->amount,
                                        'status' => true,
                                        'message' => 'success',
                                        'hash' => $getTransaction->hash
                                    ];
                                    $postData = SeamlessController::postData(
                                        $getTransaction->sites->seamless_url,
                                        $parameters
                                    );

                                    if (
                                        !empty($postData['response']->status) &&
                                        ($postData['response']->status == 200 || $postData['response']->status == 201)
                                    ) {
                                        Transaction::where('id', $getTransaction->id)->update([
                                            'player_approved' => 1,
                                            'is_success' => 1,
                                            'pay_note' => 'Otomatik Onaylanmıştır!'
                                        ]);
                                    }

                                    return response()->json([
                                        'valid' => 1,
                                        'validCode' => 211,
                                        'validTitle' => 'Kontrol ediliyor',
                                        'validMessage' => 'Yatırımınız kontrol ediliyor lütfen bekleyiniz...',
                                        'validClass' => 'info'
                                    ]);
                                }
                            }
                        }

                        return response()->json([
                            'valid' => 1,
                            'validCode' => 211,
                            'validTitle' => 'Kontrol ediliyor',
                            'validMessage' => 'Yatırımınız kontrol ediliyor lütfen bekleyiniz...',
                            'validClass' => 'info'
                        ]);
                    } else {
                        return response()->json([
                            'valid' => 1,
                            'validCode' => 211,
                            'validTitle' => 'Kontrol ediliyor',
                            'validMessage' => 'Yatırımınız kontrol ediliyor lütfen bekleyiniz...',
                            'validClass' => 'info'
                        ]);
                    }
                } else {
                    $updateTransaction = Transaction::where('id', $getTransaction->id)->update([
                        'player_approved' => 1
                    ]);
                }

                return response()->json([
                    'valid' => 1,
                    'validCode' => 211,
                    'validTitle' => 'Kontrol ediliyor',
                    'validMessage' => 'Yatırımınız kontrol ediliyor lütfen bekleyiniz...',
                    'validClass' => 'info'
                ]);
            }

            // Payment Declined
            if ($request->input('action') == 'desist') {
                $updateTransaction = Transaction::where('id', $getTransaction->id)->update([
                    'is_success' => 2,
                    'player_approved' => 2,
                    'pay_note' => 'Kullanıcı İptal Etmiştir !'
                ]);

                $parameters = [
                    'apiId' => $getTransaction->sites->id,
                    'apiKey' => $getTransaction->sites->code,
                    'paymentId' => $getTransaction->id,
                    'externalTransactionId' => $getTransaction->external_transaction_id,
                    'userId' => $getTransaction->player->external_user_id,
                    'username' => $getTransaction->player->username,
                    'type' => 'deposit',
                    'amount' => $getTransaction->amount,
                    'status' => false,
                    'message' => 'success',
                    'hash' => $getTransaction->hash
                ];
                $postData = SeamlessController::postData($getTransaction->sites->seamless_url, $parameters);

                return response()->json([
                    'valid' => 1,
                    'validCode' => 252,
                    'validTitle' => 'İşleminiz İptal Edildi',
                    'validMessage' => 'Siteye geri dönmek ister misiniz ?',
                    'validClass' => 'error'
                ]);
            }

            $getBankAccount = BankAccount::where('id', $getTransaction->share_bank_account_id)
                ->where('status', 1)
                ->first();

            if (empty($getBankAccount)) {
                $updateTransaction = Transaction::where('id', $getTransaction->id)->update([
                    'is_success' => 2,
                    'player_approved' => 2,
                    'pay_note' => 'Hesap Bilgisi Değiştiği İçin İptal Etmiştir !'
                ]);

                return response()->json([
                    'valid' => 1,
                    'validCode' => 252,
                    'validTitle' => 'İşleminiz Zaman Aşımına Uğradığı İçin İptal Edildi',
                    'validMessage' => 'Siteye geri dönmek ister misiniz ?',
                    'validClass' => 'error'
                ]);

                /*return response()->json([
                    'valid' => 1,
                    'validCode' => 205,
                    'validTitle' => 'Lütfen bekleyiniz...',
                    'validMessage' => 'İşleminiz için uygun papara hesap bilgisi hazırlanıyor... Lütfen sayfadan ayrılmayınız...',
                    'validClass' => 'info',
                    'base64Data' => ''
                ]);*/
            }

            $htmlFormat = '
            <h6 class="alert alert-danger">İŞLEM SÜRESİNCE <u>KESİNLİKLE</u> SAYFADAN AYRILMAYINIZ.</h6>
            <p>Lütfen aşağıdaki Papara hesabına <strong>' . $getTransaction->amount . ' TL</strong> yatırım gerçekleştiriniz...</p>
            <p>Hesap sahibi: <strong>' . $getBankAccount->account_owner . '</strong></p>
            <p>Hesap No: <strong>' . $getBankAccount->account_number . '</strong></p>
            <hr>
            <p><b>QR Okutarak daha hızlı para transferi yapabilirsiniz</b></p>
            <p><img src="https://chart.googleapis.com/chart?chs=200x200&chld=M|0&cht=qr&chl=https://www.papara.com/personal/qr?accountNumber=' . $getBankAccount->account_number . '%26amount=' . $getTransaction->amount . '" /></p>
            <hr>
            <p style="color:red;">Lütfen yatırım yapmadan <strong>Yatırımı gerçekleştirdim</strong> butonuna tıklamayınız...</p>
            <div class="row">
                <div class="col-md-6 mb-1 mt-1">
                    <a onclick="PaparaCheckDesist()" href="javascript:;" class="btn btn-danger btn-block" style="color:white;">Vazgeç!</a>
                </div>
                <div class="col-md-6 mb-1 mt-1">
                    <a onclick="PaparaCheckDid()" href="javascript:;" class="btn btn-success btn-block" style="color:white;">Yatırımı gerçekleştirdim!</a>
                </div>
            </div>
        ';

            $json = [
                'valid' => 1,
                'validCode' => 200,
                'validTitle' => 'Yatırım bilgileri',
                'validMessage' => 'Lütfen aşağıdaki Banka hesabına Havale/EFT ile yatırım gerçekleştiriniz...',
                'validClass' => 'success',
                'base64Data' => base64_encode($htmlFormat)
            ];

            return response()->json($json);
        } else {
            // error
        }
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function createDepositBankTransferByApi(Request $request)
    {
        $methodId = 1;
        $apiId = $request->input('api_id');
        $apiKey = $request->input('api_key');
        $playerId = $request->input('user_id');
        $username = $request->input('username');
        $firstName = $request->input('first_name');
        $lastName = $request->input('last_name');
        $email = $request->input('email');
        $phoneNumber = $request->input('phone_number');
        $lang = $request->input('lang');
        $currency = $request->input('currency');
        $externalTransactionId = $request->input('external_transaction_id');
        $returnUrlSuccess = $request->input('return_url_success');
        $returnUrlFail = $request->input('return_url_fail');

        # Check IP Address
        $checkIpAddress = WhiteListIp::checkIpAddress();
        if (!$checkIpAddress)
            return response()->json([
                'code' => 1012,
                'type' => 'error',
                'message' => trans('pay.error_1012'),
                'ip_address' => WhiteListIp::getIpAddress()
            ]);

        # Check System Status
        if (env('SYSTEM_STATUS') != 1)
            return response()->json([
                'code' => 1012,
                'type' => 'error',
                'message' => trans('pay.error_1014')
            ]);

        # Lang Dedection
        if (in_array($lang, $this->supportLang)) {
            app()->setLocale($lang);
        } else {
            app()->setLocale('en');

            return response()->json([
                'code' => 1010,
                'type' => 'error',
                'message' => trans('pay.error_1010')
            ]);
        }

        # Currency Dedection
        if (!in_array($currency, $this->supportCurrency)) {
            app()->setLocale('en');

            return response()->json([
                'code' => 1011,
                'type' => 'error',
                'message' => trans('pay.error_1011')
            ]);
        }

        if (
            empty($apiId) ||
            empty($apiKey) ||
            empty($methodId) ||
            empty($playerId) ||
            empty($username) ||
            empty($firstName) ||
            empty($lastName) ||
            empty($lang) ||
            empty($currency)
        )
            return response()->json([
                'code' => 1001,
                'type' => 'error',
                'message' => trans('pay.error_1001')
            ]);

        $checkApi = Site::where('id', $apiId)->where('code', $apiKey)->first();

        if (empty($checkApi))
            return response()->json([
                'code' => 1002,
                'type' => 'error',
                'message' => trans('pay.error_1002')
            ]);

        if ($checkApi->status != 1)
            return response()->json([
                'code' => 1003,
                'type' => 'error',
                'message' => trans('pay.error_1003')
            ]);

        # Update Return URL Success
        if (!empty($returnUrlSuccess) && $checkApi->return_url != $returnUrlSuccess) {
            Site::where('id', $checkApi->id)->update([
                'return_url' => $returnUrlSuccess
            ]);
        }

        # Update Return URL Fail
        if (!empty($returnUrlFail) && $checkApi->return_url_fail != $returnUrlFail) {
            Site::where('id', $checkApi->id)->update([
                'return_url_fail' => $returnUrlFail
            ]);
        }

        $checkPlayer = Player::where('api_id', $checkApi->id)->where('external_user_id', $playerId)->first();

        if (!empty($checkPlayer)) {
            if ($checkPlayer->status != 1)
                return response()->json([
                    'code' => 1004,
                    'type' => 'error',
                    'message' => trans('pay.error_1004')
                ]);
        }

        if (empty($checkPlayer)) {
            $insertPlayer = Player::insertGetId([
                'api_id' => $apiId,
                'username' => $username,
                'external_user_id' => $playerId,
                'external_password' => '',
                'email' => !empty($email) ? $email : null,
                'first_name' => $firstName,
                'last_name' => $lastName,
                'phone_number' => !empty($phoneNumber) ? $phoneNumber : null,
                'status' => 1,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now()
            ]);

            if ($insertPlayer) {
                $checkPlayer = Player::where('api_id', $checkApi->id)->where('external_user_id', $playerId)->first();
            } else {
                return response()->json([
                    'code' => 1004,
                    'type' => 'error',
                    'message' => trans('pay.error_1004')
                ]);
            }
        }

        $checkExternalTransaction = Transaction::where('api_id', $checkApi->id)
            ->where('external_transaction_id', $externalTransactionId)
            ->count();

        if ($checkExternalTransaction > 0)
            return response()->json([
                'code' => 1009,
                'type' => 'error',
                'message' => trans('pay.error_1009')
            ]);

        # Generate Hash
        $randomNumber = rand(1000,999999);
        $generateHash = hash("sha512", http_build_query([
            $apiId,
            $apiKey,
            $playerId,
            $username,
            !empty($externalTransactionId) ? $externalTransactionId : $checkPlayer->id . $randomNumber
        ]));

        # Create Transaction
        $privateCode = time();
        $transactionData = [
            'api_id' => $checkApi->id,
            'transaction_type' => $methodId,
            'bank_account_id' => 0,
            'external_transaction_id' => !empty($externalTransactionId) ? $externalTransactionId : '',
            'transaction_send_data' => json_encode($request->all()),
            'player_id' => $checkPlayer->id,
            'amount' => 0,
            'is_success' => 0,
            'user_id' => 0,
            'log_id' => $privateCode,
            'customer_note' => '',
            'hash' => $generateHash,
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now()
        ];

        $paymentMethod = PaymentMethod::where('status', 1)->where('id', $methodId)->first();

        if (empty($paymentMethod))
            return response()->json([
                'code' => 1005,
                'type' => 'error',
                'message' => trans('pay.error_1005')
            ]);

        $checkTransaction = Transaction::where('api_id', $checkApi->id)
            ->where('player_id', $checkPlayer->id)
            ->where('amount', '>', 0)
            ->whereIn('is_success', [0, 3])
            ->where('transaction_type', $paymentMethod->id)
            ->count();

        if ($checkTransaction > 0)
            return response()->json([
                'code' => 1007,
                'type' => 'error',
                'message' => trans('pay.error_1007')
            ]);

        $transactionId = Transaction::insertGetId($transactionData);

        if ($transactionId) {
            if ($checkApi->environment == 1) {
                $payName = 'pay';
            } else {
                $payName = 'sandbox/deposit';
            }

            return response()->json([
                'code' => 200,
                'type' => 'success',
                'message' => trans('pay.success_200'),
                'url' => env('PAY_WEBSITE_URL') . '/' . $payName . '/' . $methodId . '?hash=' . $generateHash,
                'method_id' => $methodId,
                'transaction_id' => $transactionId,
                'hash' => $generateHash
            ]);
        } else {
            return response()->json([
                'code' =>  1008,
                'type' => 'error',
                'message' => trans('pay.error_1008')
            ]);
        }
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function createDepositPaParaByApi(Request $request)
    {
        $methodId = 3;
        $apiId = $request->input('api_id');
        $apiKey = $request->input('api_key');
        $playerId = $request->input('user_id');
        $username = $request->input('username');
        $firstName = $request->input('first_name');
        $lastName = $request->input('last_name');
        $email = $request->input('email');
        $phoneNumber = $request->input('phone_number');
        $lang = $request->input('lang');
        $currency = $request->input('currency');
        $externalTransactionId = $request->input('external_transaction_id');
        $returnUrlSuccess = $request->input('return_url_success');
        $returnUrlFail = $request->input('return_url_fail');
        $siteUpdateData = [];
        $playerUpdateData = [];

        # Check IP Address
        $checkIpAddress = WhiteListIp::checkIpAddress();
        if (!$checkIpAddress)
            return response()->json([
                'code' => 1012,
                'type' => 'error',
                'message' => trans('pay.error_1012'),
                'ip_address' => WhiteListIp::getIpAddress()
            ]);

        # Check System Status
        if (env('SYSTEM_STATUS') != 1)
            return response()->json([
                'code' => 1012,
                'type' => 'error',
                'message' => trans('pay.error_1014')
            ]);

        # Lang Dedection
        if (in_array($lang, $this->supportLang)) {
            app()->setLocale($lang);
        } else {
            app()->setLocale('en');

            return response()->json([
                'code' => 1010,
                'type' => 'error',
                'message' => trans('pay.error_1010')
            ]);
        }

        # Currency Dedection
        if (!in_array($currency, $this->supportCurrency)) {
            app()->setLocale('en');

            return response()->json([
                'code' => 1011,
                'type' => 'error',
                'message' => trans('pay.error_1011')
            ]);
        }

        if (
            empty($apiId) ||
            empty($apiKey) ||
            empty($methodId) ||
            empty($playerId) ||
            empty($username) ||
            empty($firstName) ||
            empty($lastName) ||
            empty($lang) ||
            empty($currency)
        )
            return response()->json([
                'code' => 1001,
                'type' => 'error',
                'message' => trans('pay.error_1001')
            ]);

        $checkApi = Site::query()->where('id', (int)$apiId)->where('code', (string)$apiKey)->first();

        if (empty($checkApi))
            return response()->json([
                'code' => 1002,
                'type' => 'error',
                'message' => trans('pay.error_1002')
            ]);

        if ($checkApi->status != 1)
            return response()->json([
                'code' => 1003,
                'type' => 'error',
                'message' => trans('pay.error_1003')
            ]);

        # Update Return URL Success
        if (!empty($returnUrlSuccess) && $checkApi->return_url != $returnUrlSuccess) {
            $siteUpdateData['return_url'] = $returnUrlSuccess;
        }

        # Update Return URL Fail
        if (!empty($returnUrlFail) && $checkApi->return_url_fail != $returnUrlFail) {
            $siteUpdateData['return_url_fail'] = $returnUrlFail;
        }

        # Update Site
        if (!empty($siteUpdateData)) {
            Site::query()
                ->where('id', (int)$checkApi->id)
                ->update($siteUpdateData);
        }

        $checkPlayer = Player::query()
            ->where('api_id', (int)$checkApi->id)
            ->where('external_user_id', (int)$playerId)
            ->first();

        if (!empty($checkPlayer)) {
            if ($firstName != $checkPlayer->first_name) {
                $playerUpdateData['first_name'] = $firstName;
            }

            if ($lastName != $checkPlayer->last_name) {
                $playerUpdateData['last_name'] = $lastName;
            }

            if (!empty($playerUpdateData)) {
                $playerUpdateData['updated_at'] = Carbon::now();
                Player::query()
                    ->where('id', $checkPlayer->id)
                    ->update($playerUpdateData);
            }

            if ($checkPlayer->status != 1)
                return response()->json([
                    'code' => 1004,
                    'type' => 'error',
                    'message' => trans('pay.error_1004')
                ]);
        }

        if (empty($checkPlayer)) {
            $insertPlayer = Player::insertGetId([
                'api_id' => $apiId,
                'username' => $username,
                'external_user_id' => $playerId,
                'external_password' => '',
                'email' => !empty($email) ? $email : null,
                'first_name' => $firstName,
                'last_name' => $lastName,
                'phone_number' => !empty($phoneNumber) ? $phoneNumber : null,
                'status' => 1,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now()
            ]);

            if ($insertPlayer) {
                $checkPlayer = Player::query()
                    ->where('api_id', (int)$checkApi->id)
                    ->where('external_user_id', (int)$playerId)
                    ->first();
            } else {
                return response()->json([
                    'code' => 1004,
                    'type' => 'error',
                    'message' => trans('pay.error_1004')
                ]);
            }
        }

        $checkExternalTransaction = Transaction::query()
            ->where('api_id', (int)$checkApi->id)
            ->where('external_transaction_id', (string)$externalTransactionId)
            ->count();

        if ($checkExternalTransaction > 0)
            return response()->json([
                'code' => 1009,
                'type' => 'error',
                'message' => trans('pay.error_1009')
            ]);

        $checkExternalTransaction = Transaction::query()
            ->where('api_id', (int)$checkApi->id)
            ->where('player_id', (int)$playerId)
            ->where('transaction_type', 3)
            ->where('is_success', 0)
            ->count();

        if ($checkExternalTransaction > 0)
            return response()->json([
                'code' => 1009,
                'type' => 'error',
                'message' => trans('pay.error_1009')
            ]);

        # Generate Hash
        $randomNumber = rand(1000,999999);
        $generateHash = hash("sha512", http_build_query([
            $apiId,
            $apiKey,
            $playerId,
            $username,
            !empty($externalTransactionId) ? $externalTransactionId : $checkPlayer->id . $randomNumber
        ]));

        # Create Transaction
        $privateCode = time();
        $transactionData = [
            'api_id' => $checkApi->id,
            'transaction_type' => $methodId,
            'bank_account_id' => 0,
            'external_transaction_id' => !empty($externalTransactionId) ? $externalTransactionId : '',
            'transaction_send_data' => json_encode($request->all()),
            'player_id' => $checkPlayer->id,
            'amount' => 0,
            'is_success' => 0,
            'user_id' => 0,
            'log_id' => $privateCode,
            'customer_note' => '',
            'hash' => $generateHash,
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now()
        ];

        $paymentMethod = PaymentMethod::query()->where('id', $methodId)->first();

        if (empty($paymentMethod))
            return response()->json([
                'code' => 1005,
                'type' => 'error',
                'message' => trans('pay.error_1005')
            ]);

        if ($paymentMethod->status != 1)
            return response()->json([
                'code' => 1005,
                'type' => 'error',
                'message' => trans('pay.error_1005')
            ]);

        /*$checkTransaction = Transaction::where('api_id', $checkApi->id)
            ->where('player_id', $checkPlayer->id)
            ->where('amount', '>', 0)
            ->whereIn('is_success', [0, 3])
            ->where('transaction_type', $paymentMethod->id)
            ->count();

        if ($checkTransaction > 0)
            return response()->json([
                'code' => 1007,
                'type' => 'error',
                'message' => trans('pay.error_1007')
            ]);*/

        $checkTransaction = Transaction::query()
            ->where('api_id', (int)$checkApi->id)
            ->where('player_id', (int)$checkPlayer->id)
            ->where('amount', '>', 0)
            //->whereIn('is_success', [0, 3])
            ->where(function ($query) {
                $query->where('is_success', 0)
                    ->orWhere('is_success', 3);
            })
            ->where('transaction_type', (int)$paymentMethod->id)
            ->orderBy('id', 'DESC')
            ->first();

        if (!empty($checkTransaction)) {
            if ($checkApi->environment == 1) {
                $payName = 'pay';
            } else {
                $payName = 'sandbox/deposit';
            }

            return response()->json([
                'code' => 200,
                'type' => 'success',
                'message' => trans('pay.success_200'),
                'url' => env('PAY_WEBSITE_URL') . '/' . $payName . '/' . $methodId . '?hash=' . $checkTransaction->hash,
                'method_id' => $methodId,
                'transaction_id' => $checkTransaction->id,
                'hash' => $checkTransaction->hash
            ]);
        }

        $transactionId = Transaction::insertGetId($transactionData);

        if ($transactionId) {
            if ($checkApi->environment == 1) {
                $payName = 'pay';
            } else {
                $payName = 'sandbox/deposit';
            }

            return response()->json([
                'code' => 200,
                'type' => 'success',
                'message' => trans('pay.success_200'),
                'url' => env('PAY_WEBSITE_URL') . '/' . $payName . '/' . $methodId . '?hash=' . $generateHash,
                'method_id' => $methodId,
                'transaction_id' => $transactionId,
                'hash' => $generateHash
            ]);
        } else {
            return response()->json([
                'code' =>  1008,
                'type' => 'error',
                'message' => trans('pay.error_1008')
            ]);
        }
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function createDepositLobbyByApi(Request $request)
    {
        $methodId = 3;
        $apiId = $request->input('api_id');
        $apiKey = $request->input('api_key');
        $playerId = $request->input('user_id');
        $username = $request->input('username');
        $firstName = $request->input('first_name');
        $lastName = $request->input('last_name');
        $email = $request->input('email');
        $phoneNumber = $request->input('phone_number');
        $lang = $request->input('lang');
        $currency = $request->input('currency');
        $externalTransactionId = $request->input('external_transaction_id');
        $returnUrlSuccess = $request->input('return_url_success');
        $returnUrlFail = $request->input('return_url_fail');

        # Check IP Address
        $checkIpAddress = WhiteListIp::checkIpAddress();
        if (!$checkIpAddress)
            return response()->json([
                'code' => 1012,
                'type' => 'error',
                'message' => trans('pay.error_1012'),
                'ip_address' => WhiteListIp::getIpAddress()
            ]);

        # Check System Status
        if (env('SYSTEM_STATUS') != 1)
            return response()->json([
                'code' => 1012,
                'type' => 'error',
                'message' => trans('pay.error_1014')
            ]);

        # Lang Dedection
        if (in_array($lang, $this->supportLang)) {
            app()->setLocale($lang);
        } else {
            app()->setLocale('en');

            return response()->json([
                'code' => 1010,
                'type' => 'error',
                'message' => trans('pay.error_1010')
            ]);
        }

        # Currency Dedection
        if (!in_array($currency, $this->supportCurrency)) {
            app()->setLocale('en');

            return response()->json([
                'code' => 1011,
                'type' => 'error',
                'message' => trans('pay.error_1011')
            ]);
        }

        if (
            empty($apiId) ||
            empty($apiKey) ||
            empty($methodId) ||
            empty($playerId) ||
            empty($username) ||
            empty($firstName) ||
            empty($lastName) ||
            empty($lang) ||
            empty($currency)
        )
            return response()->json([
                'code' => 1001,
                'type' => 'error',
                'message' => trans('pay.error_1001')
            ]);

        $checkApi = Site::where('id', $apiId)->where('code', $apiKey)->first();

        if (empty($checkApi))
            return response()->json([
                'code' => 1002,
                'type' => 'error',
                'message' => trans('pay.error_1002')
            ]);

        if ($checkApi->status != 1)
            return response()->json([
                'code' => 1003,
                'type' => 'error',
                'message' => trans('pay.error_1003')
            ]);

        # Update Return URL Success
        if (!empty($returnUrlSuccess) && $checkApi->return_url != $returnUrlSuccess) {
            Site::where('id', $checkApi->id)->update([
                'return_url' => $returnUrlSuccess
            ]);
        }

        # Update Return URL Fail
        if (!empty($returnUrlFail) && $checkApi->return_url_fail != $returnUrlFail) {
            Site::where('id', $checkApi->id)->update([
                'return_url_fail' => $returnUrlFail
            ]);
        }

        $checkPlayer = Player::where('api_id', $checkApi->id)->where('external_user_id', $playerId)->first();

        if (!empty($checkPlayer)) {
            if ($firstName != $checkPlayer->first_name) {
                Player::where('id', $checkPlayer->id)->update([
                    'first_name' => $firstName,
                    'updated_at' => Carbon::now()
                ]);
            }

            if ($lastName != $checkPlayer->last_name) {
                Player::where('id', $checkPlayer->id)->update([
                    'last_name' => $lastName,
                    'updated_at' => Carbon::now()
                ]);
            }

            if ($checkPlayer->status != 1)
                return response()->json([
                    'code' => 1004,
                    'type' => 'error',
                    'message' => trans('pay.error_1004')
                ]);
        }

        if (empty($checkPlayer)) {
            $insertPlayer = Player::insertGetId([
                'api_id' => $apiId,
                'username' => $username,
                'external_user_id' => $playerId,
                'external_password' => '',
                'email' => !empty($email) ? $email : null,
                'first_name' => $firstName,
                'last_name' => $lastName,
                'phone_number' => !empty($phoneNumber) ? $phoneNumber : null,
                'status' => 1,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now()
            ]);

            if ($insertPlayer) {
                $checkPlayer = Player::where('api_id', $checkApi->id)->where('external_user_id', $playerId)->first();
            } else {
                return response()->json([
                    'code' => 1004,
                    'type' => 'error',
                    'message' => trans('pay.error_1004')
                ]);
            }
        }

        $checkExternalTransaction = Transaction::where('api_id', $checkApi->id)
            ->where('external_transaction_id', $externalTransactionId)
            ->count();

        if ($checkExternalTransaction > 0)
            return response()->json([
                'code' => 1009,
                'type' => 'error',
                'message' => trans('pay.error_1009')
            ]);

        # Generate Hash
        $randomNumber = rand(1000,999999);
        $generateHash = hash("sha512", http_build_query([
            $apiId,
            $apiKey,
            $playerId,
            $username,
            !empty($externalTransactionId) ? $externalTransactionId : $checkPlayer->id . $randomNumber
        ]));

        # Create Transaction
        $privateCode = time();
        $transactionData = [
            'api_id' => $checkApi->id,
            'transaction_type' => $methodId,
            'bank_account_id' => 0,
            'external_transaction_id' => !empty($externalTransactionId) ? $externalTransactionId : '',
            'transaction_send_data' => json_encode($request->all()),
            'player_id' => $checkPlayer->id,
            'amount' => 0,
            'is_success' => 0,
            'user_id' => 0,
            'log_id' => $privateCode,
            'customer_note' => '',
            'hash' => $generateHash,
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now()
        ];

        $paymentMethod = PaymentMethod::where('status', 1)->where('id', $methodId)->first();

        if (empty($paymentMethod))
            return response()->json([
                'code' => 1005,
                'type' => 'error',
                'message' => trans('pay.error_1005')
            ]);

        $checkTransaction = Transaction::where('api_id', $checkApi->id)
            ->where('player_id', $checkPlayer->id)
            ->where('amount', '>', 0)
            ->whereIn('is_success', [0, 3])
            ->where('transaction_type', $paymentMethod->id)
            ->count();

        if ($checkTransaction > 0)
            return response()->json([
                'code' => 1007,
                'type' => 'error',
                'message' => trans('pay.error_1007')
            ]);

        $transactionId = Transaction::insertGetId($transactionData);

        if ($transactionId) {
            if ($checkApi->environment == 1) {
                $payName = 'pay';
            } else {
                $payName = 'sandbox/deposit';
            }

            return response()->json([
                'code' => 200,
                'type' => 'success',
                'message' => trans('pay.success_200'),
                'url' => env('PAY_WEBSITE_URL') . '/' . $payName . '?hash=' . $generateHash,
                'method_id' => $methodId,
                'transaction_id' => $transactionId,
                'hash' => $generateHash
            ]);
        } else {
            return response()->json([
                'code' =>  1008,
                'type' => 'error',
                'message' => trans('pay.error_1008')
            ]);
        }
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function createDrawBankTransferByApi(Request $request)
    {
        $transactionPrice = $request->input('amount');
        $methodId = 7;
        $apiId = $request->input('api_id');
        $apiKey = $request->input('api_key');
        $playerId = $request->input('user_id');
        $username = $request->input('username');
        $firstName = $request->input('first_name');
        $lastName = $request->input('last_name');
        $accountNumber = $request->input('account_number');
        $bankId = $request->input('bank_id');
        $iban = $request->input('iban');
        $branchCode = $request->input('branch_code');
        $externalTransactionId = $request->input('external_transaction_id');
        $email = $request->input('email');
        $phoneNumber = $request->input('phone_number');
        $lang = $request->input('lang');
        $currency = $request->input('currency');
        $returnUrlSuccess = $request->input('return_url_success');
        $returnUrlFail = $request->input('return_url_fail');

        # Check IP Address
        $checkIpAddress = WhiteListIp::checkIpAddress();
        if (!$checkIpAddress)
            return response()->json([
                'code' => 1012,
                'type' => 'error',
                'message' => trans('pay.error_1012'),
                'ip_address' => WhiteListIp::getIpAddress()
            ]);

        # Check System Status
        if (env('SYSTEM_STATUS') != 1)
            return response()->json([
                'code' => 1012,
                'type' => 'error',
                'message' => trans('pay.error_1014')
            ]);

        # Lang Dedection
        if (in_array($lang, $this->supportLang)) {
            app()->setLocale($lang);
        } else {
            app()->setLocale('en');

            return response()->json([
                'code' => 1010,
                'type' => 'error',
                'message' => trans('pay.error_1010')
            ]);
        }

        # Currency Dedection
        if (!in_array($currency, $this->supportCurrency)) {
            app()->setLocale('en');

            return response()->json([
                'code' => 1011,
                'type' => 'error',
                'message' => trans('pay.error_1011')
            ]);
        }

        if (
            empty($transactionPrice) ||
            empty($methodId) ||
            empty($playerId) ||
            empty($username) ||
            empty($firstName) ||
            empty($lastName) ||
            empty($accountNumber) ||
            empty($bankId) ||
            empty($iban) ||
            empty($branchCode) ||
            empty($lang) ||
            empty($currency)
        )
            return response()->json([
                'code' => 1001,
                'type' => 'error',
                'message' => trans('pay.error_1001')
            ]);

        $checkApi = Site::where('id', $apiId)->where('code', $apiKey)->first();

        if (empty($checkApi))
            return response()->json([
                'code' => 1002,
                'type' => 'error',
                'message' => trans('pay.error_1002')
            ]);

        if ($checkApi->status != 1)
            return response()->json([
                'code' => 1003,
                'type' => 'error',
                'message' => trans('pay.error_1003')
            ]);

        # Update Return URL Success
        if (!empty($returnUrlSuccess) && $checkApi->return_url != $returnUrlSuccess) {
            Site::where('id', $checkApi->id)->update([
                'return_url' => $returnUrlSuccess
            ]);
        }

        # Update Return URL Fail
        if (!empty($returnUrlFail) && $checkApi->return_url_fail != $returnUrlFail) {
            Site::where('id', $checkApi->id)->update([
                'return_url_fail' => $returnUrlFail
            ]);
        }

        $checkPlayer = Player::where('api_id', $checkApi->id)->where('external_user_id', $playerId)->first();

        if (!empty($checkPlayer)) {
            if ($checkPlayer->status != 1)
                return response()->json([
                    'code' => 1004,
                    'type' => 'error',
                    'message' => trans('pay.error_1004')
                ]);
        }

        if (empty($checkPlayer)) {
            $insertPlayer = Player::insertGetId([
                'api_id' => $apiId,
                'username' => $username,
                'external_user_id' => $playerId,
                'external_password' => '',
                'email' => !empty($email) ? $email : null,
                'first_name' => $firstName,
                'last_name' => $lastName,
                'phone_number' => !empty($phoneNumber) ? $phoneNumber : null,
                'status' => 1,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now()
            ]);

            if ($insertPlayer) {
                $checkPlayer = Player::where('api_id', $checkApi->id)->where('external_user_id', $playerId)->first();
            } else {
                return response()->json([
                    'code' => 1004,
                    'type' => 'error',
                    'message' => trans('pay.error_1004')
                ]);
            }
        }

        $checkExternalTransaction = Transaction::where('api_id', $checkApi->id)
            ->where('external_transaction_id', $externalTransactionId)
            ->count();

        if ($checkExternalTransaction > 0)
            return response()->json([
                'code' => 1009,
                'type' => 'error',
                'message' => trans('pay.error_1009')
            ]);

        # Generate Hash
        $randomNumber = rand(1000,999999);
        $generateHash = hash("sha512", http_build_query([
            $apiId,
            $apiKey,
            $playerId,
            $username,
            !empty($externalTransactionId) ? $externalTransactionId : $checkPlayer->id . $randomNumber
        ]));

        # Create Transaction
        $privateCode = time();
        $transactionData = [
            'api_id' => $checkApi->id,
            'transaction_type' => $methodId,
            'bank_account_id' => 0,
            'external_transaction_id' => !empty($externalTransactionId) ? $externalTransactionId : '',
            'transaction_send_data' => json_encode($request->all()),
            'player_id' => $checkPlayer->id,
            'amount' => $transactionPrice,
            'is_success' => 0,
            'user_id' => 0,
            'log_id' => $privateCode,
            'customer_note' => '',
            'player_approved' => 1,
            'hash' => $generateHash,
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now()
        ];

        $paymentMethod = PaymentMethod::where('status', 1)->where('id', $methodId)->first();

        if (empty($paymentMethod))
            return response()->json([
                'code' => 1005,
                'type' => 'error',
                'message' => trans('pay.error_1005')
            ]);

        if ($transactionPrice > $paymentMethod->max_amount)
            return response()->json([
                'code' => 1006,
                'type' => 'error',
                'message' => trans('pay.error_1006'),
                'limit' => $paymentMethod->max_amount
            ]);

        if ($transactionPrice < $paymentMethod->min_amount)
            return response()->json([
                'code' => 1006,
                'type' => 'error',
                'message' => trans('pay.error_1006'),
                'limit' => $paymentMethod->min_amount
            ]);

        $checkTransaction = Transaction::where('api_id', $checkApi->id)
            ->where('player_id', $checkPlayer->id)
            ->where('amount', '>', 0)
            ->whereIn('is_success', [0, 3])
            ->where('transaction_type', $paymentMethod->id)
            ->count();

        if ($checkTransaction > 0)
            return response()->json([
                'code' => 1007,
                'type' => 'error',
                'message' => trans('pay.error_1007')
            ]);

        $transactionId = Transaction::insertGetId($transactionData);

        if ($transactionId) {
            if ($checkApi->environment == 1) {
                $payUrl = null;
            } else {
                $payUrl = env('PAY_WEBSITE_URL') . '/sandbox/withdraw/' . $methodId . '?hash=' . $generateHash;
            }

            return response()->json([
                'code' => 200,
                'type' => 'success',
                'message' => trans('pay.success_200'),
                'url' => $payUrl,
                'method_id' => $methodId,
                'transaction_id' => $transactionId,
                'hash' => $generateHash
            ]);
        } else {
            return response()->json([
                'code' =>  1008,
                'type' => 'error',
                'message' => trans('pay.error_1008')
            ]);
        }
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function createDrawPaParaByApi(Request $request)
    {
        $transactionPrice = $request->input('amount');
        $methodId = 8;
        $apiId = $request->input('api_id');
        $apiKey = $request->input('api_key');
        $playerId = $request->input('user_id');
        $username = $request->input('username');
        $firstName = $request->input('first_name');
        $lastName = $request->input('last_name');
        $accountNumber = $request->input('account_number');
        $externalTransactionId = $request->input('external_transaction_id');
        $email = $request->input('email');
        $phoneNumber = $request->input('phone_number');
        $lang = $request->input('lang');
        $currency = $request->input('currency');
        $returnUrlSuccess = $request->input('return_url_success');
        $returnUrlFail = $request->input('return_url_fail');

        # Check IP Address
        $checkIpAddress = WhiteListIp::checkIpAddress();
        if (!$checkIpAddress)
            return response()->json([
                'code' => 1012,
                'type' => 'error',
                'message' => trans('pay.error_1012'),
                'ip_address' => WhiteListIp::getIpAddress()
            ]);

        # Check System Status
        if (env('SYSTEM_STATUS') != 1)
            return response()->json([
                'code' => 1012,
                'type' => 'error',
                'message' => trans('pay.error_1014')
            ]);

        # Lang Dedection
        if (in_array($lang, $this->supportLang)) {
            app()->setLocale($lang);
        } else {
            app()->setLocale('en');

            return response()->json([
                'code' => 1010,
                'type' => 'error',
                'message' => trans('pay.error_1010')
            ]);
        }

        # Currency Dedection
        if (!in_array($currency, $this->supportCurrency)) {
            app()->setLocale('en');

            return response()->json([
                'code' => 1011,
                'type' => 'error',
                'message' => trans('pay.error_1011')
            ]);
        }

        if (
            empty($transactionPrice) ||
            empty($methodId) ||
            empty($playerId) ||
            empty($username) ||
            empty($firstName) ||
            empty($lastName) ||
            empty($accountNumber) ||
            empty($lang) ||
            empty($currency)
        )
            return response()->json([
                'code' => 1001,
                'type' => 'error',
                'message' => trans('pay.error_1001')
            ]);

        $checkApi = Site::query()->where('id', (int)$apiId)->where('code', (string)$apiKey)->first();

        if (empty($checkApi))
            return response()->json([
                'code' => 1002,
                'type' => 'error',
                'message' => trans('pay.error_1002')
            ]);

        if ($checkApi->status != 1)
            return response()->json([
                'code' => 1003,
                'type' => 'error',
                'message' => trans('pay.error_1003')
            ]);

        # Update Return URL Success
        if (!empty($returnUrlSuccess) && $checkApi->return_url != $returnUrlSuccess) {
            $siteUpdateData['return_url'] = $returnUrlSuccess;
        }

        # Update Return URL Fail
        if (!empty($returnUrlFail) && $checkApi->return_url_fail != $returnUrlFail) {
            $siteUpdateData['return_url_fail'] = $returnUrlFail;
        }

        # Update Site
        if (!empty($siteUpdateData)) {
            Site::query()
                ->where('id', (int)$checkApi->id)
                ->update($siteUpdateData);
        }

        $checkPlayer = Player::query()
            ->where('api_id', (int)$checkApi->id)
            ->where('external_user_id', (int)$playerId)
            ->first();

        if (!empty($checkPlayer)) {
            if ($checkPlayer->status != 1)
                return response()->json([
                    'code' => 1004,
                    'type' => 'error',
                    'message' => trans('pay.error_1004')
                ]);
        }

        if (empty($checkPlayer)) {
            $insertPlayer = Player::insertGetId([
                'api_id' => $apiId,
                'username' => $username,
                'external_user_id' => $playerId,
                'external_password' => '',
                'email' => !empty($email) ? $email : null,
                'first_name' => $firstName,
                'last_name' => $lastName,
                'phone_number' => !empty($phoneNumber) ? $phoneNumber : null,
                'status' => 1,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now()
            ]);

            if ($insertPlayer) {
                $checkPlayer = Player::query()
                    ->where('api_id', (int)$checkApi->id)
                    ->where('external_user_id', (int)$playerId)
                    ->first();
            } else {
                return response()->json([
                    'code' => 1004,
                    'type' => 'error',
                    'message' => trans('pay.error_1004')
                ]);
            }
        }

        $checkExternalTransaction = Transaction::query()
            ->where('api_id', (int)$checkApi->id)
            ->where('external_transaction_id', (string)$externalTransactionId)
            ->count();

        if ($checkExternalTransaction > 0)
            return response()->json([
                'code' => 1009,
                'type' => 'error',
                'message' => trans('pay.error_1009')
            ]);

        # Generate Hash
        $randomNumber = rand(1000,999999);
        $generateHash = hash("sha512", http_build_query([
            $apiId,
            $apiKey,
            $playerId,
            $username,
            !empty($externalTransactionId) ? $externalTransactionId : $checkPlayer->id . $randomNumber
        ]));

        # Create Transaction
        $privateCode = time();
        $transactionData = [
            'api_id' => $checkApi->id,
            'transaction_type' => $methodId,
            'bank_account_id' => 0,
            'external_transaction_id' => !empty($externalTransactionId) ? $externalTransactionId : '',
            'transaction_send_data' => json_encode($request->all()),
            'player_id' => $checkPlayer->id,
            'amount' => $transactionPrice,
            'is_success' => 0,
            'user_id' => 0,
            'log_id' => $privateCode,
            'customer_note' => '',
            'hash' => $generateHash,
            'player_approved' => 1,
            'created_at' => Carbon::now(),
            'updated_at' => Carbon::now()
        ];

        $paymentMethod = PaymentMethod::query()->where('id', $methodId)->first();

        if (empty($paymentMethod))
            return response()->json([
                'code' => 1005,
                'type' => 'error',
                'message' => trans('pay.error_1005')
            ]);

        if ($paymentMethod->status != 1)
            return response()->json([
                'code' => 1005,
                'type' => 'error',
                'message' => trans('pay.error_1005')
            ]);

        if ($transactionPrice > $paymentMethod->max_amount)
            return response()->json([
                'code' => 1006,
                'type' => 'error',
                'message' => trans('pay.error_1006'),
                'limit' => $paymentMethod->max_amount
            ]);

        if ($transactionPrice < $paymentMethod->min_amount)
            return response()->json([
                'code' => 1006,
                'type' => 'error',
                'message' => trans('pay.error_1006'),
                'limit' => $paymentMethod->min_amount
            ]);

        $checkTransaction = Transaction::query()
            ->where('api_id', (int)$checkApi->id)
            ->where('player_id', (int)$checkPlayer->id)
            ->where('amount', '>', 0)
            //->whereIn('is_success', [0, 3])
            ->where(function ($query) {
                $query->where('is_success', 0)
                    ->orWhere('is_success', 3);
            })
            ->where('transaction_type', (int)$paymentMethod->id)
            ->orderBy('id', 'DESC')
            ->count();

        if ($checkTransaction > 0)
            return response()->json([
                'code' => 1007,
                'type' => 'error',
                'message' => trans('pay.error_1007')
            ]);

        $transactionId = Transaction::insertGetId($transactionData);

        if ($transactionId) {
            if ($checkApi->environment == 1) {
                $payUrl = null;
            } else {
                $payUrl = env('PAY_WEBSITE_URL') . '/sandbox/withdraw/' . $methodId . '?hash=' . $generateHash;
            }

            return response()->json([
                'code' => 200,
                'type' => 'success',
                'message' => trans('pay.success_200'),
                'url' => $payUrl,
                'method_id' => $methodId,
                'transaction_id' => $transactionId,
                'hash' => $generateHash
            ]);
        } else {
            return response()->json([
                'code' =>  1008,
                'type' => 'error',
                'message' => trans('pay.error_1008')
            ]);
        }
    }

    /**
     * @param $bankAccount
     * @return array
     */
    private function getCheckLimit($bankAccount)
    {
        $dataBankAccounts = [];

        if (!empty($bankAccount->process_limit) && $bankAccount->process_limit > 0) {
            if ($bankAccount->process_limit > Transaction::getCountProcessLimit($bankAccount->id)) {
                $dataBankAccounts = [
                    'id' => $bankAccount->id
                ];
            }
        } else {
            if (
                $bankAccount->total_limit > Transaction::getSumTotalLimit($bankAccount->id) &&
                $bankAccount->daily_limit > Transaction::getSumDailyLimit($bankAccount->id)
            ) {
                $dataBankAccounts = [
                    'id' => $bankAccount->id
                ];
            }
        }

        return $dataBankAccounts;
    }

    /**
     * @param $text
     * @return mixed|string
     */
    public function turkishStrToUpper($text)
    {
        $search = array("ç", "i", "ı", "ğ", "ö", "ş", "ü");
        $replace = array("Ç", "İ", "I", "Ğ", "Ö", "Ş", "Ü");
        $text = str_replace($search, $replace, $text);
        $text = strtoupper($text);
        return $text;
    }

    /**
     * @param $name
     * @return array
     */
    private function nameChecker($name)
    {
        $explodeName = explode(' ', $name);
        $totalName = sizeof($explodeName);

        if ($totalName > 2) {
            $lastName = $explodeName[$totalName-1];
            $firstName = '';
            unset($explodeName[$totalName-1]);
            foreach ($explodeName as $key => $value) {
                $firstName .= ' ' . $value;
            }
            $firstName = trim($firstName);
        } else {
            $firstName = $explodeName[0];
            $lastName = $explodeName[1];
        }

        return [
            'first_name' => $firstName,
            'last_name' => $lastName
        ];
    }
}
