<?php

namespace App;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;

class Transaction extends Model
{
    protected $table = 'transactions';

    public function sites()
    {
        return $this->hasOne(Site::class, 'id', 'api_id');
    }

    public function payment_method()
    {
        return $this->hasOne(PaymentMethod::class, 'id', 'transaction_type');
    }

    public function player()
    {
        return $this->hasOne(Player::class, 'id', 'player_id');
    }

    public function user()
    {
        return $this->hasOne(User::class, 'id', 'user_id');
    }

    public static function getCountProcessLimit($bankAccountId)
    {
        return self::where('bank_account_id', $bankAccountId)
            ->where('is_success', 1)
            ->count();
    }

    public static function getSumTotalLimit($bankAccountId)
    {
        return self::where('bank_account_id', $bankAccountId)
            ->where('is_success', 1)
            ->sum('amount');
    }

    public static function getSumDailyLimit($bankAccountId)
    {
        return self::where('bank_account_id', $bankAccountId)
            ->where('is_success', 1)
            ->whereDate('created_at', Carbon::today())
            ->sum('amount');
    }

    /**
     * @param $bankAccountId
     * @param $createdAt
     * @return int
     */
    public static function getCountDepositByBankAccountId($bankAccountId, $createdAt): int
    {
        return self::query()
            ->where('is_success', 1)
            ->whereIn('transaction_type', [1, 2, 3, 4, 5, 6, 9, 10])
            ->where(function ($query) use ($bankAccountId) {
                $query->where('bank_account_id', $bankAccountId)
                    ->orWhere('share_bank_account_id', $bankAccountId);
            })
            ->where('created_at', '>', $createdAt)
            ->count();
    }

    /**
     * @param $bankAccountId
     * @param $createdAt
     * @return int
     */
    public static function getCountDrawByBankAccountId($bankAccountId, $createdAt): int
    {
        return self::query()
            ->where('is_success', 1)
            ->whereIn('transaction_type', [7, 8])
            ->where(function ($query) use ($bankAccountId) {
                $query->where('bank_account_id', $bankAccountId)
                    ->orWhere('share_bank_account_id', $bankAccountId);
            })
            ->where('created_at', '>', $createdAt)
            ->count();
    }

    /**
     * @param $hash
     * @param array $data
     * @return mixed
     */
    public static function updateTransactionByHash($hash, $data = [])
    {
        return self::where('hash', $hash)->update($data);
    }
}
