<?php

namespace App\Http\Controllers;

use App\Bank;
use App\BankAccount;
use App\CronJob;
use App\DepositRange;
use App\Log;
use App\RequestLog;
use App\PaymentMethod;
use App\Player;
use App\PushBulletSeen;
use App\PushBulletSms;
use App\Site;
use App\SiteCommission;
use App\Transaction;
use App\SxTransaction;
use App\SxLog;
use App\User;
use App\WhiteListIp;
use Carbon\Carbon;
use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Routing\Redirector;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\View\View;
use Pushbullet\Push;
use App\GoogleAuthenticator;
use Jenssegers\Agent\Agent;

class PanelController extends Controller
{
    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getLogin(Request $request)
    {
        if (auth()->check())
            return redirect('/admin/dashboard');

        if ($request->method() == 'POST') {
            $required = [
                'username' => 'required',
                'password' => ['required', 'min:6', 'max:50'],
                '_token' => 'required'
            ];

            $valid = validator()->make($request->all(), $required);
            if (!$valid->fails()) {
                $username = $request->input('username');
                $password = $request->input('password');
                $agent = new Agent();

               // if (!$agent->isDesktop())
                 //   return back()->with('error', 'Giriş yapmaya çalıştığınız platform erişimi engellenmiştir.');

                if (auth()->attempt(['username' => $username, 'password' => $password, 'status' => 1])) {
                    $user = auth()->user();
                    Log::addLog($user->id, 6);

                    if ($user->ga_status == 1) {
                        User::googleAuthenticatorLoginDisable($user->id);

                        return redirect('/admin/ga_login');
                    } else {
                        User::query()
                            ->where('id', $user->id)
                            ->update([
                                'ip_address' => getIpAddress(),
                                'updated_at' => Carbon::now()
                            ]);

                        return redirect('/admin/dashboard');
                    }
                } else {
                    return back()->with('error', 'Kullanıcı Adı ya da Parola yanlıştır.');
                }
            }
        }

        return view('panel.login');
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getLoginGoogleAuthenticator(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();
        $ga = new GoogleAuthenticator();
        $secret = $user->ga_secret ?? null;

        if (empty($secret))
            return redirect('/admin/dashboard');

        if ($user->ga_status == 0)
            return redirect('/admin/dashboard');

        if ($user->ga_login == 1)
            return redirect('/admin/dashboard');

        if ($request->method() == 'POST') {
            $required = [
                'ga_code' => ['required', 'min:6', 'max:6'],
                '_token' => 'required'
            ];

            $valid = validator()->make($request->all(), $required);
            if (!$valid->fails()) {
                $gaCode = $request->input('ga_code');

                if ($ga->verifyCode($secret, $gaCode, 2)) {
                    Log::addLog($user->id, 28, 11);
                    User::googleAuthenticatorLoginEnable($user->id);

                    User::query()
                        ->where('id', $user->id)
                        ->update([
                            'ip_address' => getIpAddress(),
                            'updated_at' => Carbon::now()
                        ]);

                    return redirect('/admin/dashboard');
                } else {
                    return back()->with('error', 'Doğrulama kodu hatalıdır !');
                }
            } else {
                return back()->with('error', 'Boş alan bırakmayınız !');
            }
        }

        return view('panel.ga_login');
    }

    /**
     * @return RedirectResponse|Redirector
     */
    public function getLogout()
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        Log::addLog($user->id, 5);

        User::query()
            ->where('id', $user->id)
            ->update([
                'ip_address' => null,
                'updated_at' => Carbon::now()
            ]);

        auth()->logout();

        return redirect('/admin/login');
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getDashboard(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();
        $startDate = date('Y-m-d').' 00:00:00';
        $endDate = date('Y-m-d').' 23:59:59';
        $setDateCacheKey = !empty($user->api_id) ? 'dashboard_set_date' : 'dashboard_set_date_' . $user->api_id;
        $transactionCacheKey = !empty($user->api_id) ? 'dashboard_transactions' : 'dashboard_transactions_' . $user->api_id;
        $activeTransactionPlayerCacheKey = !empty($user->api_id) ? 'dashboard_active_transaction_players' : 'dashboard_active_transaction_players_' . $user->api_id;
        $playerCacheKey = !empty($user->api_id) ? 'dashboard_players' : 'dashboard_players_' . $user->api_id;

        if ($user->role_id == 4)
            return redirect('/admin/get-bank-accounts');

        if ($request->input('cache') == 'clear') {
            Cache::forget($setDateCacheKey);
            Cache::forget($transactionCacheKey);
            Cache::forget($activeTransactionPlayerCacheKey);
            Cache::forget($playerCacheKey);

            return redirect('/admin/dashboard');
        }

        // Transactions
        if (!Cache::has($transactionCacheKey)) {
            $getTransactions = new Transaction();
            $getTransactions = $getTransactions->query();
            $getTransactions = $getTransactions->with('payment_method');

            if (!empty($user->api_id))
                $getTransactions = $getTransactions->where('api_id', $user->api_id);

            $getTransactions = $getTransactions->where('updated_at', '>', $startDate);
            $getTransactions = $getTransactions->where('updated_at', '<', $endDate);
            $getTransactions = $getTransactions->orderBy('id', 'DESC');
            $getTransactions = $getTransactions->get();

            $dataTransactions = [];
            $activeTransactionPlayers = [];

            foreach ($getTransactions as $transaction) {
                if (empty($dataTransactions[$transaction->payment_method->name]))
                    $dataTransactions[$transaction->payment_method->name] = [
                        'paymentMethod' => [
                            'id' => $transaction->payment_method->id,
                            'name' => $transaction->payment_method->name,
                            'type' => $transaction->payment_method->type,
                        ],
                        'counts' => [
                            'deposit' => [
                                'amount' => 0,
                                'piece' => 0
                            ],
                            'withdraw' => [
                                'amount' => 0,
                                'piece' => 0
                            ]
                        ]
                    ];

                if ($transaction->is_success == 1) {
                    if ($transaction->payment_method->type == 1) {
                        $dataTransactions[$transaction->payment_method->name]['counts']['deposit']['amount'] += $transaction->amount;
                        $dataTransactions[$transaction->payment_method->name]['counts']['deposit']['piece']++;
                        $activeTransactionPlayers[$transaction->player_id] = $transaction->player_id;
                    }

                    if ($transaction->payment_method->type == 2) {
                        $dataTransactions[$transaction->payment_method->name]['counts']['withdraw']['amount'] += $transaction->amount;
                        $dataTransactions[$transaction->payment_method->name]['counts']['withdraw']['piece']++;
                    }
                }
            }

            $cacheDate = Carbon::now();
            Cache::put($transactionCacheKey, $dataTransactions, 300);
            Cache::put($activeTransactionPlayerCacheKey, $activeTransactionPlayers, 300);
            Cache::put($setDateCacheKey, $cacheDate, 300);
        } else {
            $activeTransactionPlayers = Cache::get($activeTransactionPlayerCacheKey);
            $dataTransactions = Cache::get($transactionCacheKey);
            $cacheDate = Cache::get($setDateCacheKey);
        }

        // Players
        if (!Cache::has($playerCacheKey)) {
            $activePlayer = new Player();
            $activePlayer = $activePlayer->query();

            if (!empty($user->api_id))
                $activePlayer = $activePlayer->where('api_id', $user->api_id);

            $activePlayer = $activePlayer->where('created_at', '>', $startDate);
            $activePlayer = $activePlayer->where('created_at', '<', $endDate);
            $activePlayer = $activePlayer->where('status', 1);
            $activePlayer = $activePlayer->count();

            Cache::put($playerCacheKey, $activePlayer, 300);
        } else {
            $activePlayer = Cache::get($playerCacheKey);
        }

        return view('panel.dashboard')
            ->with('user', $user)
            ->with('transactions', $dataTransactions)
            ->with('activeTransactionPlayers', count($activeTransactionPlayers))
            ->with('activePlayer', $activePlayer)
            ->with('cacheDate', $cacheDate);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getBanks(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        $getBanks = Bank::where('status', 1)->orderBy('id', 'DESC')->get();

        return view('panel.banks')
            ->with('user', $user)
            ->with('banks', $getBanks);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getBankCreate(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        if ($request->method() == 'POST') {
            $bankName = $request->input('bank_name');
            $mobileBank = $request->input('mobile_bank');
            $param = $request->input('param');
            $status = $request->input('status');

            $insertData = Bank::insertGetId([
                'name' => $bankName,
                'mobile_bank' => $mobileBank,
                'param' => $param,
                'status' => $status
            ]);

            if ($insertData) {
                Log::addLog($user->id, 17, 7, $insertData);

                return redirect('/admin/account-management/banks');
            } else {
                return back()->with('error', 'Hata, kayıt oluşturulamadı !');
            }
        }

        return view('panel.create_bank')
            ->with('user', $user);
    }

    /**
     * @param $id
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getBankEdit($id, Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        $getBank = Bank::where('id', $id)->first();

        if (empty($getBank))
            return redirect('/admin/account-management/banks');

        if ($request->method() == 'POST') {
            $bankName = $request->input('bank_name');
            $mobileBank = $request->input('mobile_bank');
            $param = $request->input('param');
            $status = $request->input('status');

            $updateData = Bank::where('id', $id)->update([
                'name' => $bankName,
                'mobile_bank' => $mobileBank,
                'param' => $param,
                'status' => $status
            ]);

            if ($updateData) {
                Log::addLog($user->id, 18, 7, $id);

                return redirect('/admin/account-management/banks');
            } else {
                return back()->with('error', 'Hata, kayıt düzenlenemedi !');
            }
        }

        return view('panel.update_bank')
            ->with('user', $user)
            ->with('detail', $getBank);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getAccounts(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id == 3)
            return redirect('/admin/dashboard');

        $getBanks = Bank::where('status', 1)->get();
        $getDepositRanges = DepositRange::where('status', 1)->get();

        return view('panel.accounts')
            ->with('user', $user)
            ->with('banks', $getBanks)
            ->with('depositRange', $getDepositRanges);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getAccountList(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        $status = $request->input('status');
        $searchQuery = $request->input('q');
        $processId = $request->input('process_id');
        $typeId = $request->input('type_id');
        $bankId = $request->input('bank_id');
        $depositRangeId = $request->input('deposit_range_id');

        $getBankAccounts = new BankAccount();
        $getBankAccounts = $getBankAccounts->with('deposit_range');

        if (
            empty($processId) &&
            empty($searchQuery) &&
            empty($typeId) &&
            empty($bankId) &&
            empty($depositRangeId) &&
            $status != ''
        )
            $getBankAccounts = $getBankAccounts->where('bank_accounts.status', $status);

        if (!empty($processId))
            $getBankAccounts = $getBankAccounts->where('bank_accounts.id', $processId);

        if (!empty($typeId))
            $getBankAccounts = $getBankAccounts->where('bank_accounts.type_id', $typeId);

        if (!empty($bankId))
            $getBankAccounts = $getBankAccounts->where('bank_accounts.bank_id', $bankId);

        if (!empty($depositRangeId))
            $getBankAccounts = $getBankAccounts->where('bank_accounts.deposit_range_id', $depositRangeId);

        if (!empty($searchQuery))
            $getBankAccounts = $getBankAccounts->where('bank_accounts.account_owner', 'LIKE', '%'.$searchQuery.'%')
                ->orWhere('bank_accounts.account_number', 'LIKE', '%'.$searchQuery.'%')
                ->orWhere('bank_accounts.iban', 'LIKE', '%'.$searchQuery.'%')
                ->orWhere('bank_accounts.branch_code', 'LIKE', '%'.$searchQuery.'%')
                ->orWhere('banks.name', 'LIKE', '%'.$searchQuery.'%');

        $getBankAccounts = $getBankAccounts->leftJoin('banks', function($join) {
            $join->on('bank_accounts.bank_id','=','banks.id')
                ->where('bank_accounts.bank_id', '!=', 0);
        });

        $getBankAccounts = $getBankAccounts->select(
            'bank_accounts.*',
            'banks.name as bank_name'
        );
        $getBankAccounts = $getBankAccounts->orderBy('banks.name', 'ASC');
        $getBankAccounts = $getBankAccounts->orderBy('bank_accounts.id', 'DESC');
        $getBankAccounts = $getBankAccounts->paginate(20);

        $dataSites = [];
        $getSites = Site::where('status', 1)->get();

        foreach ($getSites as $site) {
            $dataSites[$site->id] = [
                'id' => $site->id,
                'name' => $site->name
            ];
        }

        return view('panel.account_list')
            ->with('user', $user)
            ->with('bankAccounts', $getBankAccounts)
            ->with('sites', $dataSites);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getAccountCreate(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id == 3)
            return redirect('/admin/dashboard');

        $getBanks = Bank::where('status', 1)->get();
        $getDepositRange = DepositRange::where('status', 1)->get();
        $getSites = Site::where('status', 1)->get();

        if ($request->method() == 'POST') {
            $typeId = $request->input('type_id');
            $bankId = $request->input('bank_id');
            $accountNumber = $request->input('account_number');
            $branchCode = $request->input('branch_code');
            $accountOwner = $request->input('account_owner');
            $iban = $request->input('iban');
            $totalLimit = $request->input('total_limit');
            $dailyLimit = $request->input('daily_limit');
            $processLimit = $request->input('process_limit');
            $description = $request->input('description');
            $adminNote = $request->input('admin_note');
            $status = $request->input('status');
            $depositRangeId = $request->input('deposit_range_id');
            $mailServerCheck = $request->input('mail_server_check');
            $mailServer = $request->input('mail_server');
            $mailUsername = $request->input('mail_username');
            $mailPassword = $request->input('mail_password');
            $apiId = $request->input('api_id');
            $paParaIban = (int)$request->input('papara_iban') ?? 0;

            if (!empty($mailServer) && $mailServerCheck != 1)
                return back()->with('error', 'Hata, mail server test edilmemiştir !');

            $insertData = BankAccount::insertGetId([
                'type_id' => $typeId,
                'bank_id' => $bankId,
                'account_number' => $accountNumber,
                'branch_code' => $branchCode,
                'account_owner' => $accountOwner,
                'iban' => $iban,
                'balance' => 0,
                'total_limit' => $totalLimit,
                'daily_limit' => $dailyLimit,
                'process_limit' => $processLimit,
                'description' => $description,
                'admin_note' => $adminNote,
                'status' => $status,
                'is_pp_iban' => $paParaIban,
                'api_id' => !empty($apiId) ? implode(',', $apiId) . ',' : null,
                'deposit_range_id' => !empty($depositRangeId) ? $depositRangeId : null,
                'mail_server' => !empty($mailServer) ? $mailServer : null,
                'mail_username' => !empty($mailUsername) ? $mailUsername : null,
                'mail_password' => !empty($mailPassword) ? $mailPassword : null
            ]);

            if ($insertData) {
                Log::addLog($user->id, 21, 9, $insertData);

                return redirect('/admin/account-management/accounts');
            } else {
                return back()->with('error', 'Hata, kayıt oluşturulamadı !');
            }
        }

        return view('panel.create_account')
            ->with('user', $user)
            ->with('banks', $getBanks)
            ->with('deposit_range', $getDepositRange)
            ->with('sites', $getSites);
    }

    /**
     * @param $id
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getAccountEdit($id, Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id == 3)
            return redirect('/admin/dashboard');

        $getAccount = BankAccount::where('id', $id)->first();
        $getBanks = Bank::where('status', 1)->get();
        $getDepositRange = DepositRange::where('status', 1)->get();
        $getSites = Site::where('status', 1)->get();

        if (empty($getAccount))
            return redirect('/admin/account-management/accounts');

        if ($request->method() == 'POST') {
            $typeId = $request->input('type_id');
            $bankId = $request->input('bank_id');
            $accountNumber = $request->input('account_number');
            $branchCode = $request->input('branch_code');
            $accountOwner = $request->input('account_owner');
            $iban = $request->input('iban');
            $totalLimit = $request->input('total_limit');
            $dailyLimit = $request->input('daily_limit');
            $processLimit = $request->input('process_limit');
            $description = $request->input('description');
            $adminNote = $request->input('admin_note');
            $balance = $request->input('balance');
            $status = $request->input('status');
            $depositRangeId = $request->input('deposit_range_id');
            $mailServerCheck = $request->input('mail_server_check');
            $mailServer = $request->input('mail_server');
            $mailUsername = $request->input('mail_username');
            $mailPassword = $request->input('mail_password');
            $apiId = $request->input('api_id');
            $paParaIban = (int)$request->input('papara_iban') ?? 0;

            if (!empty($mailServer) && $mailServerCheck != 1)
                return back()->with('error', 'Hata, mail server test edilmemiştir !');

            $updateData = BankAccount::where('id', $id)->update([
                'type_id' => $typeId,
                'bank_id' => $bankId,
                'account_number' => $accountNumber,
                'branch_code' => $branchCode,
                'account_owner' => $accountOwner,
                'iban' => $iban,
                'balance' => $balance,
                'total_limit' => $totalLimit,
                'daily_limit' => $dailyLimit,
                'process_limit' => $processLimit,
                'description' => $description,
                'admin_note' => $adminNote,
                'status' => $status,
                'is_pp_iban' => $paParaIban,
                'api_id' => !empty($apiId) ? implode(',', $apiId) . ',' : null,
                'deposit_range_id' => !empty($depositRangeId) ? $depositRangeId : null,
                'mail_server' => !empty($mailServer) ? $mailServer : null,
                'mail_username' => !empty($mailUsername) ? $mailUsername : null,
                'mail_password' => !empty($mailPassword) ? $mailPassword : null
            ]);

            if ($updateData) {
                Log::addLog($user->id, 22, 9, $id);

                return redirect('/admin/account-management/accounts');
            } else {
                return back()->with('error', 'Hata, kayıt düzenlenemedi !');
            }
        }

        return view('panel.update_account')
            ->with('user', $user)
            ->with('detail', $getAccount)
            ->with('banks', $getBanks)
            ->with('deposit_range', $getDepositRange)
            ->with('sites', $getSites);
    }

    /**
     * @param Request $request
     * @return JsonResponse|RedirectResponse|Redirector
     */
    public function getAccountMailCheck(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id == 3)
            return redirect('/admin/dashboard');

        if ($request->method() == 'POST') {
            $mailServer = $request->input('mail_server');
            $mailUsername = $request->input('mail_username');
            $mailPassword = $request->input('mail_password');

            if (empty($mailServer))
                return response()->json(['status' => false, 'message' => 'Lütfen bir servis sağlayıcı seçiniz !']);

            if (empty($mailUsername))
                return response()->json(['status' => false, 'message' => 'Lütfen bir email adresi giriniz !']);

            if (empty($mailPassword))
                return response()->json(['status' => false, 'message' => 'Lütfen email şifrenizi giriniz !']);

            $path = BankAccount::getMailServer($mailServer);
            $connect = PaParaController::connect($path, $mailUsername, $mailPassword, 0, 1);

            if ($connect['status'] == true && empty($connect['errors'])) {
                return response()->json([
                    'status' => true,
                    'message' => 'Test Başarılı, Kullanılabilir!'
                ]);
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'Mail Server bağlanamadı ! Lütfen bu konuda yardım almak için sistem yetkilisi ile görüşünüz.'
                ]);
            }
        }

        return response()->json([
            'status' => false,
            'message' => 'Lütfen, POST parametresi ile işlem yapınız.'
        ]);
    }

    /**
     * @param $id
     * @param $status
     * @param Request $request
     * @return RedirectResponse|Redirector
     */
    public function setAccountStatus($id, $status, Request $request)
    {
        if (!auth()->check())
            return response()->json([
                'type' => 'error',
                'message' => 'Lütfen, üye girişi yapınız !'
            ]);

        $user = auth()->user();

        if ($user->role_id == 3)
            return response()->json([
                'type' => 'error',
                'message' => 'Hata, yetkiniz bulunmamaktadır !'
            ]);

        $getAccount = BankAccount::where('id', $id)->first();

        if (empty($getAccount))
            return response()->json([
                'type' => 'error',
                'message' => 'Hata, kayıt bulunamadı !'
            ]);

        $updateData = BankAccount::where('id', $id)->update([
            'status' => $status,
        ]);

        if ($updateData) {
            Log::addLog($user->id, 23, 9, $id);

            return response()->json([
                'type' => 'success',
                'message' => 'Kayıt için durum değiştirilmiştir.'
            ]);
        } else {
            return response()->json([
                'type' => 'error',
                'message' => 'Hata, kayıt durumu düzenlenmedi !'
            ]);
        }
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getSites(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();
        #echo '<pre>';
        #print_r(SxLog::query()->orderBy('id', 'DESC')->limit(100)->get()); die;
        //print_r(SxTransaction::query()->orderBy('id', 'DESC')->limit(100)->get()); die;

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        $getSites = Site::where('status', 1)->orderBy('id', 'DESC')->get();

        return view('panel.sites')
            ->with('user', $user)
            ->with('sites', $getSites);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getSiteCreate(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        if ($request->method() == 'POST') {
            $name = $request->input('name');
            $code = $request->input('code');
            $seamlessUrl = $request->input('seamless_url');
            $platform = $request->input('platform');
            $environment = $request->input('environment');
            $status = $request->input('status');
            $returnUrl = $request->input('return_url');
            $returnUrlFail = $request->input('return_url_fail');

            $checkSiteCode = Site::where('code', $code)->get();

            if (count($checkSiteCode) > 0)
                return back()->with('error', 'Hata, belirtilen site kodu mevcuttur.');

            if (empty($seamlessUrl))
                return back()->with('error', 'Hata, seamless url alanı boş olamaz.');

            $insertData = Site::insertGetId([
                'name' => $name,
                'code' => $code,
                'seamless_url' => $seamlessUrl,
                'platform' => $platform,
                'environment' => $environment,
                'status' => $status,
                'return_url' => $returnUrl,
                'return_url_fail' => $returnUrlFail
            ]);

            if ($insertData) {
                Log::addLog($user->id, 13, 1, $insertData);

                return redirect('/admin/sites');
            } else {
                return back()->with('error', 'Hata, kayıt oluşturulamadı !');
            }
        }

        return view('panel.create_site')
            ->with('user', $user);
    }

    /**
     * @param $id
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getSiteEdit($id, Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        $getSite = Site::query()->where('id', $id)->first();

        if (empty($getSite))
            return redirect('/admin/sites');

        if ($request->method() == 'POST' && $request->input('method') == 'api') {
            $name = $request->input('name');
            $seamlessUrl = $request->input('seamless_url');
            $platform = $request->input('platform');
            $environment = $request->input('environment');
            $status = $request->input('status');
            $returnUrl = $request->input('return_url');
            $returnUrlFail = $request->input('return_url_fail');
            $isBtType = $request->input('is_bt_type');
            $isPType = $request->input('is_p_type');
            $isPPType = $request->input('is_pp_type');

            if (empty($seamlessUrl))
                return back()->with('error', 'Hata, seamless url alanı boş olamaz.');

            $updateData = Site::query()
                ->where('id', $id)
                ->update([
                    'name' => $name,
                    'seamless_url' => $seamlessUrl,
                    'platform' => $platform,
                    'environment' => $environment,
                    'status' => $status,
                    'return_url' => $returnUrl,
                    'return_url_fail' => $returnUrlFail,
                    'is_bt_type' => $isBtType,
                    'is_p_type' => $isPType,
                    'is_pp_type' => $isPPType
                ]);

            if ($updateData) {
                Log::addLog($user->id, 14, 1, $id);

                return redirect('/admin/sites');
            } else {
                return back()->with('error', 'Hata, kayıt düzenlenemedi !');
            }
        }

        if ($request->method() == 'POST' && $request->input('method') == 'commission') {
            $errors = 0;
            $getAllParams = $request->except([
                '_token',
                'method'
            ]);

            foreach ($getAllParams as $key => $value) {
                if (strpos($key, 'payment_status_') !== false) {
                    $replaceId = str_replace('payment_status_', '', $key);
                }

                if (strpos($key, 'commission_') !== false) {
                    $replaceId = str_replace('commission_', '', $key);
                    $checkCommission = SiteCommission::where('site_id', $id)
                        ->where('payment_method_id', $replaceId)
                        ->first();

                    # Check Value Not Null
                    if (empty($value))
                        $value = 0;

                    # Check Value (,) Operator Clear
                    $value = str_replace(',', '.', $value);

                    if (!empty($checkCommission)) {
                        $updateCommission = SiteCommission::where('id', $checkCommission->id)->update([
                            'percent' => $value,
                            'updated_at' => Carbon::now()
                        ]);

                        if ($updateCommission) {
                            Log::addLog($user->id, 25, 1, $id);
                        } else {
                            $errors += 1;
                        }
                    } else {
                        $insertCommission = SiteCommission::insertGetId([
                            'site_id' => $id,
                            'payment_method_id' => $replaceId,
                            'percent' => $value,
                            'created_at' => Carbon::now(),
                            'updated_at' => Carbon::now()
                        ]);

                        if ($insertCommission) {
                            Log::addLog($user->id, 24, 1, $id);
                        } else {
                            $errors += 1;
                        }
                    }
                }
            }

            if ($errors == 0) {
                return redirect('/admin/sites');
            } else {
                return back()->with('error', 'Hata, kayıt düzenlenemedi !');
            }
        }

        $getPaymentMethods = PaymentMethod::query()->where('status', 1)->get();

        return view('panel.update_site')
            ->with('user', $user)
            ->with('detail', $getSite)
            ->with('paymentMethods', $getPaymentMethods);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getOperators(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        if ($request->input('status') == '0')
            $status = 0;
        else
            $status = 1;

        $getOperators = User::orderBy('id', 'DESC')->get();

        return view('panel.operators')
            ->with('user', $user)
            ->with('operators', $getOperators)
            ->with('status', $status);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getBankAccountsForWebSiteAdmin(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 3 && $user->role_id != 4)
            return redirect('/admin/dashboard');

        $getActiveBankAccounts = BankAccount::where('status', 1)->get();

        $dataBankAccounts = [];
        foreach ($getActiveBankAccounts as $bankAccount) {
            if (!empty($bankAccount->process_limit) && $bankAccount->process_limit > 0) {
                if ($bankAccount->process_limit > Transaction::getCountProcessLimit($bankAccount->id)) {
                    $dataBankAccounts[] = $bankAccount;
                }
            } else {
                if (
                    $bankAccount->total_limit > Transaction::getSumTotalLimit($bankAccount->id) &&
                    $bankAccount->daily_limit > Transaction::getSumDailyLimit($bankAccount->id)
                ) {
                    $dataBankAccounts[] = $bankAccount;
                }
            }
        }

        return view('panel.get_bank_accounts')
            ->with('user', $user)
            ->with('bankAccounts', $dataBankAccounts);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getOperatorCreate(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        if ($request->method() == 'POST') {
            $apiId = $request->input('api_id');
            $username = $request->input('username');
            $password = $request->input('password');
            $email = $request->input('email');
            $firstName = $request->input('first_name');
            $lastName = $request->input('last_name');
            $roleId = $request->input('role_id');
            $status = $request->input('status');
            $accessContact = $request->input('access_contact');
            $accessManuelTransaction = $request->input('access_manuel_transaction');
            $accessPushBullet = $request->input('access_push_bullet');

            if (empty($username) || empty($password) || empty($email) || empty($firstName) || empty($lastName))
                return back()->with('error', 'Hata, boş alan bırakmayınız.');

            if (empty($apiId) && $roleId == 3)
                return back()->with('error', 'Hata, lütfen site seçiniz.');

            $checkUsername = User::where('username', $username)->get();

            if (count($checkUsername) > 0)
                return back()->with('error', 'Hata, belirtilen kullanıcı adı mevcuttur.');

            $checkEmail = User::where('email', $email)->get();

            if (count($checkEmail) > 0)
                return back()->with('error', 'Hata, belirtilen email mevcuttur.');

            $insertData = User::insertGetId([
                'api_id' => !empty($apiId) ? $apiId : null,
                'username' => $username,
                'password' => bcrypt($password),
                'email' => $email,
                'first_name' => $firstName,
                'last_name' => $lastName,
                'role_id' => $roleId,
                'access_contact' => $accessContact,
                'access_manuel_transaction' => $accessManuelTransaction,
                'access_push_bullet' => $accessPushBullet,
                'status' => $status
            ]);

            if ($insertData) {
                Log::addLog($user->id, 15, 6, $insertData);

                return redirect('/admin/operators');
            } else {
                return back()->with('error', 'Hata, kayıt oluşturulamadı !');
            }
        }

        $getSites = Site::where('status', 1)->get();

        return view('panel.create_operator')
            ->with('user', $user)
            ->with('sites', $getSites);
    }

    /**
     * @param $id
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getOperatorEdit($id, Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        $getOperator = User::where('id', $id)->first();

        if (empty($getOperator))
            return redirect('/admin/operators');

        if ($request->method() == 'POST') {
            $apiId = $request->input('api_id');
            $username = $request->input('username');
            $password = $request->input('password');
            $email = $request->input('email');
            $firstName = $request->input('first_name');
            $lastName = $request->input('last_name');
            $roleId = $request->input('role_id');
            $status = $request->input('status');
            $accessContact = $request->input('access_contact');
            $accessManuelTransaction = $request->input('access_manuel_transaction');
            $accessPushBullet = $request->input('access_push_bullet');

            if (empty($username) || empty($email) || empty($firstName) || empty($lastName))
                return back()->with('error', 'Hata, boş alan bırakmayınız.');

            if (empty($apiId) && $roleId == 3)
                return back()->with('error', 'Hata, lütfen site seçiniz.');

            if ($getOperator->username != $username) {
                $checkUsername = User::where('username', $username)->get();

                if (count($checkUsername) > 0)
                    return back()->with('error', 'Hata, belirtilen kullanıcı adı mevcuttur.');
            }

            if ($getOperator->email != $email) {
                $checkEmail = User::where('email', $email)->get();

                if (count($checkEmail) > 0)
                    return back()->with('error', 'Hata, belirtilen email mevcuttur.');
            }

            $newData = [
                'api_id' => !empty($apiId) ? $apiId : null,
                'username' => $username,
                'email' => $email,
                'first_name' => $firstName,
                'last_name' => $lastName,
                'role_id' => $roleId,
                'access_contact' => $accessContact,
                'access_manuel_transaction' => $accessManuelTransaction,
                'access_push_bullet' => $accessPushBullet,
                'status' => $status
            ];

            if (!empty($password))
                $newData['password'] = bcrypt($password);

            $updateData = User::where('id', $id)->update($newData);

            if ($updateData) {
                Log::addLog($user->id, 16, 6, $id);

                return redirect('/admin/operators');
            } else {
                return back()->with('error', 'Hata, kayıt düzenlenemedi !');
            }
        }

        $getSites = Site::where('status', 1)->get();

        return view('panel.update_operator')
            ->with('user', $user)
            ->with('detail', $getOperator)
            ->with('sites', $getSites);
    }

    /**
     * @param $id
     * @return RedirectResponse|Redirector
     */
    public function disableOperatorGoogleAuthenticator($id)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        $getOperator = User::where('id', $id)->first();

        if (empty($getOperator))
            return redirect('/admin/operators/ga/' . $id);

        $updateOperator = User::where('id', $id)->update([
            'ga_status' => 0,
            'ga_login' => 0,
            'ga_secret' => null,
            'updated_at' => Carbon::now()
        ]);

        if ($updateOperator) {
            Log::addLog($user->id, 29, 11, $id);

            return redirect('/admin/operators/ga/' . $id);
        } else
            return back()->with('error', 'Google Authenticator kapatılırken bir hata oluştu !');
    }

    /**
     * @param $id
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getOperatorGoogleAuthenticator($id, Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        $getOperator = User::where('id', $id)->first();

        if (empty($getOperator))
            return redirect('/admin/operators');

        if ($request->method() == 'POST') {
            $newData = [
                'ga_status' => 1,
                'updated_at' => Carbon::now()
            ];

            $updateData = User::where('id', $id)->update($newData);

            if ($updateData) {
                Log::addLog($user->id, 30, 11, $id);

                return redirect('/admin/operators/ga/' . $id);
            } else {
                return back()->with('error', 'Google Authenticator aktifleştirilirken bir hata oluştu !');
            }
        }

        $ga = new GoogleAuthenticator();
        $secret = $getOperator->ga_secret ?? $ga->createSecret();

        $accountName = env('PAY_WEBSITE_NAME') . ' - ' . $getOperator->username;
        $qrCodeUrl = $ga->getQRCodeGoogleUrl($accountName, $secret);

        # Update G2A Secret for User
        User::query()
            ->where('id', $getOperator->id)
            ->update([
                'ga_secret' => $secret,
                'updated_at' => Carbon::now()
            ]);

        return view('panel.ga_operator')
            ->with('user', $user)
            ->with('detail', $getOperator)
            ->with('qrCodeUrl', $qrCodeUrl)
            ->with('accountName', $accountName)
            ->with('accountSecret', $secret);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getTransactions(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id == 4)
            return redirect('/admin/dashboard');

        $getPaymentMethods = PaymentMethod::query()->where('status', 1)->get();
        $getBanks = Bank::query()->where('status', 1)->get();
        $getDepositRanges = DepositRange::query()->where('status', 1)->get();
        $getOperators = User::query()
            ->where('status', 1)
            ->where('role_id', '!=', 3)
            ->orderBy('id', 'DESC')
            ->get();

        $getSites = new Site();
        $getSites = $getSites->query();

        if (!empty($user->api_id))
            $getSites = $getSites->where('id', $user->api_id);

        $getSites = $getSites->where('status', 1);
        $getSites = $getSites->get();

        return view('panel.transactions')
            ->with('user', $user)
            ->with('sites', $getSites)
            ->with('banks', $getBanks)
            ->with('paymentMethods', $getPaymentMethods)
            ->with('depositRanges', $getDepositRanges)
            ->with('operators', $getOperators);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getTransactionList(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        $typeId = $request->input('type_id');
        $status = $request->input('status');
        $methodId = $request->input('method_id');
        $bankId = $request->input('bank_id');
        $apiId = $request->input('api_id');
        $transactionId = $request->input('transaction_id');
        $minAmount = $request->input('min_amount');
        $maxAmount = $request->input('max_amount');
        $depositRangeId = $request->input('deposit_range_id');
        $playerQ = $request->input('player_q');
        $processUserId = $request->input('process_user_id');
        #$startDate = date('Y-m-d', strtotime('-3 days')) . ' 00:00:00';
        #$endDate = date('Y-m-d H:i:s');

        if ($request->method() == 'POST') {
            if (!empty($request->input('filter_start_date')))
                $startDate = $request->input('filter_start_date') . ':00';

            if (!empty($request->input('filter_end_date')))
                $endDate = $request->input('filter_end_date') . ':59';
        }

        # Payment Methods List
        $paymentMethod = new PaymentMethod();
        $paymentMethod = $paymentMethod->query();

        if (!empty($typeId))
            $paymentMethod = $paymentMethod->where('type', $typeId);

        $paymentMethod = $paymentMethod->get();

        # Payment Method to Array
        $payments = [];
        $payments_in = [];
        foreach ($paymentMethod as $plist) {
            $payments[$plist->id] = $plist;
            $payments_in[] = $plist->id;
        }

        # Bank Accounts List
        if (!empty($bankId) || !empty($depositRangeId)) {
            $bankAccount = new BankAccount();
            $bankAccount = $bankAccount->query();

            if (!empty($bankId)) {
                $bankAccount = $bankAccount->where('bank_id', (int)$bankId);
                $bankAccount = $bankAccount->where('type_id', 1);
            }

            if (!empty($depositRangeId))
                $bankAccount = $bankAccount->where('deposit_range_id', (int)$depositRangeId);

            /*$bankAccount = $bankAccount->where(function ($query) use ($bankId, $depositRangeId) {
                $query->where('bank_id', (int)$bankId)
                    ->orWhere('deposit_range_id', (int)$depositRangeId);
            });*/

            $bankAccount = $bankAccount->get();

            # Payment Method to Array
            $bankAccounts = [];
            $bankAccounts_in = [];
            foreach ($bankAccount as $plist) {
                $bankAccounts[$plist->id] = $plist;
                $bankAccounts_in[] = $plist->id;
            }
        }

        # Player List
        if (!empty($playerQ)) {
            $playerQExplode = playerQueryExplode($playerQ);

            $getPlayers = new Player();
            $getPlayers = $getPlayers->query();

            $getPlayers = $getPlayers->where(function ($query) use ($playerQ, $playerQExplode) {
                if ($playerQExplode['length'] >= 2) {
                    $query->where('first_name', 'LIKE', '%' . $playerQExplode['first_name'] . '%')
                        ->where('last_name', 'LIKE', '%' . $playerQExplode['last_name'] . '%');
                } else {
                    $query->where('id', $playerQ)
                        ->orWhere('external_user_id', $playerQ)
                        ->orWhere('username', 'LIKE', '%' . $playerQ . '%')
                        ->orWhere('sx_match_name', 'LIKE', '%' . $playerQ . '%')
                        ->orWhere('first_name', 'LIKE', '%' . $playerQ . '%')
                        ->orWhere('last_name', 'LIKE', '%' . $playerQ . '%');
                }
            });

            $getPlayers = $getPlayers->get();

            # Players to Array
            $playersIn = [];
            foreach ($getPlayers as $player) {
                $playersIn[] = $player->id;
            }
        }

        $getTransactions = new Transaction();
        $getTransactions = $getTransactions->with(['payment_method', 'player', 'sites']);

        if (!empty($transactionId))
            $getTransactions = $getTransactions->where('id', (int)$transactionId);
        else {
            if (!empty($startDate))
                $getTransactions = $getTransactions->where('updated_at', '>', $startDate);

            if (!empty($endDate))
                $getTransactions = $getTransactions->where('updated_at', '<', $endDate);

            if ($status != '')
                $getTransactions = $getTransactions->where('is_success', (int)$status);

            if (!empty($bankId) || !empty($depositRangeId)) {
                #$getTransactions = $getTransactions->whereIn('bank_account_id', $bankAccounts_in);

                if (!empty($bankId))
                    $getTransactions = $getTransactions->where(function ($query) use ($bankAccounts_in, $bankId) {
                        $query->whereIn('bank_account_id', $bankAccounts_in)
                            ->orWhere('transaction_send_data', 'LIKE', '%"bank_id":"' . $bankId . '"%');
                    });

                if (!empty($depositRangeId))
                    $getTransactions = $getTransactions->where(function ($query) use ($bankAccounts_in) {
                        $query->whereIn('share_bank_account_id', $bankAccounts_in)
                            ->orWhereIn('bank_account_id', $bankAccounts_in);
                    });;
            }

            if (!empty($playerQ)) {
                $getTransactions = $getTransactions->whereIn('player_id', $playersIn);
            }

            $getTransactions = $getTransactions->whereIn('transaction_type', $payments_in);
            $getTransactions = $getTransactions->where('amount', '>', 0);
            $getTransactions = $getTransactions->where('player_approved', '!=', 0);

            if (!empty($methodId))
                $getTransactions = $getTransactions->where('transaction_type', (int)$methodId);

            if (!empty($apiId))
                $getTransactions = $getTransactions->where('api_id', (int)$apiId);

            if (!empty($processUserId))
                $getTransactions = $getTransactions->where('user_id', (int)$processUserId);

            if (!empty($minAmount) && $minAmount > 0)
                $getTransactions = $getTransactions->where('amount', '>=', $minAmount);

            if (!empty($maxAmount) && $maxAmount > 0)
                $getTransactions = $getTransactions->where('amount', '<=', $maxAmount);
        }

        $getTransactions = $getTransactions->orderBy('updated_at', 'DESC');
        $getTransactions = $getTransactions->orderBy('id', 'DESC');
        $getTransactions = $getTransactions->paginate(20);
        #$getTransactions = $getTransactions->get();
        #echo $getTransactions->toSql(); die;

        return view('panel.transaction_list')
            ->with('user', $user)
            ->with('transactions', $getTransactions);
    }

    /**
     * @param Request $request
     * @return Application|RedirectResponse|Redirector|void
     */
    public function exportExcelTransactions(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $filename = "data_export_" . date('Ymd') . ".xls";
        header("Content-Type: application/vnd.ms-excel");
        header("Content-Disposition: attachment; filename=\"$filename\"");

        $user = auth()->user();

        # Params Decode
        parse_str($request->input('params'), $params);

        $typeId = $params['type_id'];
        $status = $params['status'];
        $methodId = $params['method_id'];
        $bankId = $params['bank_id'];
        $apiId = $params['api_id'];
        $transactionId = $params['transaction_id'];
        $minAmount = $params['min_amount'];
        $maxAmount = $params['max_amount'];
        $depositRangeId = $params['deposit_range_id'];
        $playerQ = $params['player_q'];
        $processUserId = $params['process_user_id'];
        $startDate = date('Y-m-d', strtotime('-3 days')) . ' 00:00:00';
        $endDate = date('Y-m-d H:i:s');

        if ($request->method() == 'POST') {
            if (!empty($params['filter_start_date']))
                $startDate = $params['filter_start_date'] . ':00';

            if (!empty($params['filter_end_date']))
                $endDate = $params['filter_end_date'] . ':59';
        }

        if (dayDifferenceCalculation($startDate, $endDate) > 31) {
            echo "Filtreleme kısmında bulunan 2 tarih arasındaki fark maksimum 1 ay olmalıdır! \n";
            exit();
            die;
        }

        # Payment Methods List
        $paymentMethod = new PaymentMethod();
        $paymentMethod = $paymentMethod->query();

        if (!empty($typeId))
            $paymentMethod = $paymentMethod->where('type', $typeId);

        $paymentMethod = $paymentMethod->get();

        # Payment Method to Array
        $payments = [];
        $payments_in = [];
        foreach ($paymentMethod as $plist) {
            $payments[$plist->id] = $plist;
            $payments_in[] = $plist->id;
        }

        # Bank Accounts List
        if (!empty($bankId) || !empty($depositRangeId)) {
            $bankAccount = new BankAccount();
            $bankAccount = $bankAccount->query();

            if (!empty($bankId)) {
                $bankAccount = $bankAccount->where('bank_id', (int)$bankId);
                $bankAccount = $bankAccount->where('type_id', 1);
            }

            if (!empty($depositRangeId))
                $bankAccount = $bankAccount->where('deposit_range_id', (int)$depositRangeId);

            /*$bankAccount = $bankAccount->where(function ($query) use ($bankId, $depositRangeId) {
                $query->where('bank_id', (int)$bankId)
                    ->orWhere('deposit_range_id', (int)$depositRangeId);
            });*/

            $bankAccount = $bankAccount->get();

            # Payment Method to Array
            $bankAccounts = [];
            $bankAccounts_in = [];
            foreach ($bankAccount as $plist) {
                $bankAccounts[$plist->id] = $plist;
                $bankAccounts_in[] = $plist->id;
            }
        }

        # Player List
        if (!empty($playerQ)) {
            $playerQExplode = playerQueryExplode($playerQ);

            $getPlayers = new Player();
            $getPlayers = $getPlayers->query();

            $getPlayers = $getPlayers->where(function ($query) use ($playerQ, $playerQExplode) {
                if ($playerQExplode['length'] >= 2) {
                    $query->where('first_name', 'LIKE', '%' . $playerQExplode['first_name'] . '%')
                        ->where('last_name', 'LIKE', '%' . $playerQExplode['last_name'] . '%');
                } else {
                    $query->where('id', $playerQ)
                        ->orWhere('external_user_id', $playerQ)
                        ->orWhere('username', 'LIKE', '%' . $playerQ . '%')
                        ->orWhere('sx_match_name', 'LIKE', '%' . $playerQ . '%')
                        ->orWhere('first_name', 'LIKE', '%' . $playerQ . '%')
                        ->orWhere('last_name', 'LIKE', '%' . $playerQ . '%');
                }
            });

            $getPlayers = $getPlayers->get();

            # Players to Array
            $playersIn = [];
            foreach ($getPlayers as $player) {
                $playersIn[] = $player->id;
            }
        }

        $getTransactions = new Transaction();
        //$getTransactions = $getTransactions->with('payment_method');
        $getTransactions = $getTransactions->with(['payment_method', 'player', 'sites']);

        if (!empty($transactionId))
            $getTransactions = $getTransactions->where('id', (int)$transactionId);
        else {
            if (!empty($startDate))
                $getTransactions = $getTransactions->where('created_at', '>', $startDate);

            if (!empty($endDate))
                $getTransactions = $getTransactions->where('created_at', '<', $endDate);

            if ($status != '')
                $getTransactions = $getTransactions->where('is_success', (int)$status);

            if (!empty($bankId) || !empty($depositRangeId)) {
                #$getTransactions = $getTransactions->whereIn('bank_account_id', $bankAccounts_in);

                if (!empty($bankId))
                    $getTransactions = $getTransactions->where(function ($query) use ($bankAccounts_in, $bankId) {
                        $query->whereIn('bank_account_id', $bankAccounts_in)
                            ->orWhere('transaction_send_data', 'LIKE', '%"bank_id":"' . $bankId . '"%');
                    });

                if (!empty($depositRangeId))
                    $getTransactions = $getTransactions->where(function ($query) use ($bankAccounts_in) {
                        $query->whereIn('share_bank_account_id', $bankAccounts_in)
                            ->orWhereIn('bank_account_id', $bankAccounts_in);
                    });;
            }

            if (!empty($playerQ)) {
                $getTransactions = $getTransactions->whereIn('player_id', $playersIn);
            }

            $getTransactions = $getTransactions->whereIn('transaction_type', $payments_in);
            $getTransactions = $getTransactions->where('amount', '>', 0);
            $getTransactions = $getTransactions->where('player_approved', '!=', 0);

            if (!empty($methodId))
                $getTransactions = $getTransactions->where('transaction_type', (int)$methodId);

            if (!empty($apiId))
                $getTransactions = $getTransactions->where('api_id', (int)$apiId);

            if (!empty($processUserId))
                $getTransactions = $getTransactions->where('user_id', (int)$processUserId);

            if (!empty($minAmount) && $minAmount > 0)
                $getTransactions = $getTransactions->where('amount', '>=', $minAmount);

            if (!empty($maxAmount) && $maxAmount > 0)
                $getTransactions = $getTransactions->where('amount', '<=', $maxAmount);
        }

        $getTransactions = $getTransactions->orderBy('id', 'DESC');
        $getTransactions = $getTransactions->get();


        echo "# \t";
        echo "Site Adı \t";
        echo "Tip \t";
        echo "Oyuncu \t";
        echo "Yöntem \t";
        echo "Miktar \t";
        echo "Durum \t";
        echo "Oluşturma Tarihi \n";

        foreach ($getTransactions as $t) {
            if($t->sites->platform === 3 || $t->sites->platform === 2) {
                $playerName = $t->player->external_user_id . ' (' . $t->player->first_name . ' '. $t->player->last_name . ')';
            } else {
                $playerName = $t->player->username . ' (' . $t->player->first_name . ' '. $t->player->last_name . ')';
            }

            echo $t->id . "\t";
            echo $t->sites->name . "\t";
            echo PaymentMethod::getType($t->payment_method->type) . "\t";
            echo $playerName . "\t";
            echo $t->payment_method->name . "\t";
            echo $t->amount . "\t";
            echo getTransactionStatusName($t->is_success) . "\t";
            echo $t->created_at . "\n";
        }

        exit();
        die;
    }

    /**
     * @param $id
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getTransactionDetail($id, Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        $getTransaction = Transaction::where('id', (int)$id)->first();
        $extraParams = json_decode($getTransaction->transaction_send_data);
        $paymentMethod = $getTransaction->payment_method;
        $site = $getTransaction->sites;

        if (!empty($getTransaction->bank_account_id)) {
            $bankAccount = BankAccount::where('id', $getTransaction->bank_account_id)->first();
        } else {
            if (!empty($getTransaction->share_bank_account_id)) {
                $bankAccount = BankAccount::where('id', $getTransaction->share_bank_account_id)->first();
            } else {
                $bankAccount = [];
            }
        }

        if ($paymentMethod->id != 1 && $paymentMethod->id != 2 && $paymentMethod->id != 7)
            unset($extraParams->bank_id);

        $unsetParams = [
            'site-username',
            'site-fullname',
            'site-bank',
            'deposit-amount',
            'site-userID',
            'site-data',
            'site-trx',
            'site-id',
            'hash',
            'bank_account_id',
            '_token',
            'api_id',
            'api_key',
            'customer_note',
            'player_id',
            'method_id',
            'method-id',
            'return_url_success',
            'return_url_fail',
            'email',
            'phone_number',
            'payment_method',
            'type',
            'amount'
        ];

        foreach ($unsetParams as $up) {
            unset($extraParams->{$up});
        }

        // Eğer ki işlem beklemede değil ise ödeme yöntemlerini getir.
        if ((int)$getTransaction->is_success !== 0) {
          $getPaymentMethods = PaymentMethod::get();
        } else {
          $getPaymentMethods = null;
        }

        return view('panel.transaction_detail')
            ->with('user', $user)
            ->with('transaction', $getTransaction)
            ->with('extraParams', $extraParams)
            ->with('paymentMethod', $paymentMethod)
            ->with('paymentMethods', $getPaymentMethods)
            ->with('site', $site)
            ->with('bankAccount', $bankAccount);
    }

    /**
     * @param $id
     * @param Request $request
     * @return JsonResponse|RedirectResponse|Redirector
     */
    public function getTransactionStatus($id, Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id == 3)
            return redirect('/admin/dashboard');

        $transactionStatus = $request->input('transaction_status');
        $payNote = $request->input('pay_note');
        $getTransaction = Transaction::where('id', $id)->first();
        $extraParams = json_decode($getTransaction->transaction_send_data);
        $paymentMethod = $getTransaction->payment_method;
        $site = $getTransaction->sites;
        $paymentType = $paymentMethod->type == 1 ? 'deposit' : 'withdraw';

        # Find Bank Account Id
        if (!empty($getTransaction->share_bank_account_id)) {
            $bankAccountId = $getTransaction->share_bank_account_id;
        } else {
            if (!empty($getTransaction->bank_account_id)) {
                $bankAccountId = $getTransaction->bank_account_id;
            } else {
                $bankAccountId = null;
            }
        }

        if (empty($site->seamless_url) || $site->seamless_url == '#') {
            $transactionUpdateStatus = true;
        } else {
            if (!empty($getTransaction->hash)) {
                if ($transactionStatus == 1 || $transactionStatus == 2) {
                    if (empty($site->seamless_url))
                        return response()->json([
                            'type' => 'error',
                            'message' => 'Hata, lütfen callback adresi giriniz.'
                        ]);

                    # Seamless (CallBack) Send Request
                    $parameters = [
                        'apiId' => $site->id,
                        'apiKey' => $site->code,
                        'paymentId' => $getTransaction->id,
                        'externalTransactionId' => $getTransaction->external_transaction_id,
                        'userId' => $getTransaction->player->external_user_id,
                        'username' => $getTransaction->player->username,
                        'type' => $paymentType,
                        'amount' => $getTransaction->amount,
                        'status' => $transactionStatus == 1 ? true : false,
                        'message' => 'success',
                        'hash' => $getTransaction->hash
                    ];
                    $postData = SeamlessController::postData($site->seamless_url, $parameters);

                    # Hash Calculate
                    $hashCalculate = $this->hashCalculate(
                        $site->id,
                        $site->code,
                        $getTransaction->player->external_user_id,
                        $getTransaction->player->username,
                        $getTransaction->amount,
                        $paymentType,
                        $getTransaction->external_transaction_id
                    );

                    # Add Request Log
                    RequestLog::addLog(
                        $id,
                        $user->id,
                        (2),
                        $parameters,
                        $postData['response'],
                        $postData['time']
                    );

                    // Error correction when multiple requests are sent is for ProNet Gaming platform.
                    if ($site->platform === 3) {
                        $getRequestLog = RequestLog::query()
                            ->where('transaction_id', (int)$id)
                            ->where('type_id', 2)
                            ->orderBy('id', 'ASC')
                            ->first();

                        if (!empty($getRequestLog)) {
                            $jsonRequestContent = json_decode($getRequestLog->response_content);

                            if (!empty($jsonRequestContent) && (int)$jsonRequestContent->status === 200) {
                                $transactionUpdateStatus = true;
                            } else {
                                return response()->json([
                                    'type' => 'error',
                                    'message' => 'Hata, Pronet Gaming tarafından kaynaklı sorun mevcuttur.',
                                    'callback' => $postData['response'],
                                    'parameters' => $parameters,
                                    'hash' => $hashCalculate
                                ]);
                            }
                        } else {
                            # Check Hash Calculate
                            if (!empty($postData['response']->status)) {
                                # Check Hash Empty
                                if (empty($postData['response']->hash))
                                    return response()->json([
                                        'type' => 'error',
                                        'message' => 'Hata, callback tarafından gelen hash bulunmamaktadır.',
                                        'callback' => $postData['response'],
                                        'parameters' => $parameters,
                                        'hash' => $hashCalculate
                                    ]);

                                # Check Hash Calculate
                                if (!$this->checkHash($postData['response']->hash, $hashCalculate))
                                    return response()->json([
                                        'type' => 'error',
                                        'message' => 'Hata, callback tarafından gelen hash geçersizdir.',
                                        'callback' => $postData['response'],
                                        'parameters' => $parameters,
                                        'hash' => $hashCalculate
                                    ]);
                            }

                            if (!empty($postData['response']->status) && $postData['response']->status == 200) {
                                $transactionUpdateStatus = true;
                            } else if (!empty($postData['response']->status) && $postData['response']->status == 201) {
                                $transactionUpdateStatus = true;
                            } else {
                                $transactionUpdateStatus = false;
                            }
                        }
                    } else {
                        # Check Hash Calculate
                        if (!empty($postData['response']->status)) {
                            # Check Hash Empty
                            if (empty($postData['response']->hash))
                                return response()->json([
                                    'type' => 'error',
                                    'message' => 'Hata, callback tarafından gelen hash bulunmamaktadır.',
                                    'callback' => $postData['response'],
                                    'parameters' => $parameters,
                                    'hash' => $hashCalculate
                                ]);

                            # Check Hash Calculate
                            if (!$this->checkHash($postData['response']->hash, $hashCalculate))
                                return response()->json([
                                    'type' => 'error',
                                    'message' => 'Hata, callback tarafından gelen hash geçersizdir.',
                                    'callback' => $postData['response'],
                                    'parameters' => $parameters,
                                    'hash' => $hashCalculate
                                ]);
                        }

                        if (!empty($postData['response']->status) && $postData['response']->status == 200) {
                            $transactionUpdateStatus = true;
                        } else if (!empty($postData['response']->status) && $postData['response']->status == 201) {
                            $transactionUpdateStatus = true;
                        } else {
                            $transactionUpdateStatus = false;
                        }
                    }
                } else {
                    $transactionUpdateStatus = true;
                }
            } else {
                $transactionUpdateStatus = true;
            }
        }

        if ($transactionUpdateStatus) {
            $updateTransaction = Transaction::where('id', $id)->update([
                'is_success' => $transactionStatus,
                'pay_note' => $payNote,
                'user_id' => $user->id
            ]);
        } else {
            $updateTransaction = false;
        }

        if ($updateTransaction) {
            # If the status is not in progress.
            if ($transactionStatus != 3) {
                if ($paymentMethod->type == 1) {
                    if ($transactionStatus == 1)
                        BankAccount::addBankBalance($bankAccountId, $getTransaction->amount);
                } else if ($paymentMethod->type == 2) {
                    if ($transactionStatus == 1)
                        BankAccount::removeBankBalance($bankAccountId, $getTransaction->amount);
                }
            }

            Log::addLog($user->id, 9, 4, $id);

            return response()->json([
                'type' => 'success',
                'message' => 'Talebiniz başarıyla sonuçlandırıldı.'
            ]);
        } else {
            return response()->json([
                'type' => 'error',
                'message' => 'Hata, lütfen tekrar deneyiniz.',
                'callback' => $postData['response']
            ]);
        }
    }

    /**
     * @return JsonResponse|RedirectResponse|Redirector
     */
    public function checkAutoTransaction()
    {
      return response()->json([
          'type' => 'success',
          'message' => 'success'
      ]);

        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if (env('PAPARA_TYPE') == 3) {
            $getTransactions = Transaction::where('is_success', 0)
                ->where('transaction_type', 3)
                ->whereNotNull('hash')
                ->get();

            foreach ($getTransactions as $transaction) {
                $nowDate = Carbon ::now();
                $transactionDate = Carbon ::parse($transaction->created_at);

                $diffInMinutes = $transactionDate->diffInMinutes($nowDate, false);

                if ($diffInMinutes > 1 && $diffInMinutes < 7) {
                    $payController = new PayController();
                    $siteFullName = $payController->turkishStrToUpper($transaction->player->first_name . ' ' . $transaction->player->last_name);
                    $getBankAccountDetails = BankAccount::where('id', $transaction->share_bank_account_id)->first();
                    $mailPath = BankAccount::getMailServer($getBankAccountDetails->mail_server);
                    $connect = PaParaController::connect(
                        $mailPath,
                        $getBankAccountDetails->mail_username,
                        $getBankAccountDetails->mail_password,
                        1,
                        20
                    );

                    if ($connect['status'] == true && !empty($connect['data'])) {
                        foreach ($connect['data'] as $key => $value) {
                            if ($value['status'] == true) {
                                $autoSender = $this->turkishStrToUpper($value['sender']);
                                $autoAmount = $value['money'];

                                # Find Transaction
                                if ($siteFullName == $autoSender && $transaction->amount == $autoAmount) {
                                    if (empty($transaction->sites->seamless_url) || $transaction->sites->seamless_url == '#') {
                                        Transaction::where('id', $transaction->id)->update([
                                            'player_approved' => 1,
                                            'is_success' => 1,
                                            'pay_note' => 'Sistem Tarafından Onaylanmıştır!'
                                        ]);

                                        Log::addLog($user->id, 10, 4, $transaction->id);
                                    } else {
                                        # Seamless (CallBack) Send Request
                                        $parameters = [
                                            'apiId' => $transaction->sites->id,
                                            'apiKey' => $transaction->sites->code,
                                            'paymentId' => $transaction->id,
                                            'externalTransactionId' => $transaction->external_transaction_id,
                                            'userId' => $transaction->player->external_user_id,
                                            'username' => $transaction->player->username,
                                            'type' => 'deposit',
                                            'amount' => $transaction->amount,
                                            'status' => true,
                                            'message' => 'success',
                                            'hash' => $transaction->hash
                                        ];
                                        $postData = SeamlessController::postData(
                                            $transaction->sites->seamless_url,
                                            $parameters
                                        );

                                        if (
                                            !empty($postData['response']->status) &&
                                            ($postData['response']->status == 200 || $postData['response']->status == 201)
                                        ) {
                                            Transaction::where('id', $transaction->id)->update([
                                                'player_approved' => 1,
                                                'is_success' => 1,
                                                'pay_note' => 'Sistem Tarafından Onaylanmıştır!'
                                            ]);

                                            Log::addLog($user->id, 10, 4, $transaction->id);
                                        }
                                    }
                                }
                            }
                        }
                    }
                } else if ($diffInMinutes > 7) {
                    if (empty($transaction->sites->seamless_url) || $transaction->sites->seamless_url == '#') {
                        Transaction::where('id', $transaction->id)->update([
                            'is_success' => 2,
                            'amount' => $transaction->amount == 0 ? 50 : $transaction->amount,
                            'pay_note' => 'Sistem Tarafından İptal Edilmiştir!'
                        ]);

                        Log::addLog($user->id, 10, 4, $transaction->id);
                    } else {
                        # Seamless (CallBack) Send Request
                        $parameters = [
                            'apiId' => $transaction->sites->id,
                            'apiKey' => $transaction->sites->code,
                            'paymentId' => $transaction->id,
                            'externalTransactionId' => $transaction->external_transaction_id,
                            'userId' => $transaction->player->external_user_id,
                            'username' => $transaction->player->username,
                            'type' => 'deposit',
                            'amount' => $transaction->amount,
                            'status' => false,
                            'message' => 'success',
                            'hash' => $transaction->hash
                        ];
                        $postData = SeamlessController::postData(
                            $transaction->sites->seamless_url,
                            $parameters
                        );

                        if (
                            !empty($postData['response']->status) &&
                            ($postData['response']->status == 200 || $postData['response']->status == 201)
                        ) {
                            Transaction::where('id', $transaction->id)->update([
                                'is_success' => 2,
                                'amount' => $transaction->amount == 0 ? 50 : $transaction->amount,
                                'pay_note' => 'Sistem Tarafından İptal Edilmiştir!'
                            ]);

                            Log::addLog($user->id, 10, 4, $transaction->id);
                        }
                    }
                } else {
                    // waiting...
                }
            }
        }


        return response()->json([
            'type' => 'success',
            'message' => 'success'
        ]);
    }

    /**
     * @return JsonResponse|RedirectResponse|Redirector
     */
    public function getTransactionsCount()
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        $getTransactions = new Transaction();
        $getTransactions = $getTransactions->with('payment_method');

        if (!empty($user->api_id))
            $getTransactions = $getTransactions->where('api_id', $user->api_id);

        $getTransactions = $getTransactions->where('amount', '>', 0);
        //$getTransactions = $getTransactions->whereIn('is_success', [0,3]);
        $getTransactions = $getTransactions->where(function ($query) {
            $query->where('is_success', 0)->orWhere('is_success', 3);
        });
        $getTransactions = $getTransactions->where('player_approved', '!=', 0);
        $getTransactions = $getTransactions->get();
        $responseData = [
            'deposit' => 0,
            'withdraw' => 0,
            'in_progress' => 0
        ];
        $isControl = 0;
        $countTransactions = [
            'new' => 0,
            'pending' => 0,
        ];
        foreach ($getTransactions as $transaction) {
            if ($transaction->payment_method->type == 1)
                $responseData['deposit']++;

            if ($transaction->payment_method->type == 2)
                $responseData['withdraw']++;

            if ($transaction->is_success == 0)
                $countTransactions['new']++;

            if ($transaction->is_success == 3)
                $countTransactions['pending']++;

            if ($transaction->is_success == 3)
                $responseData['in_progress']++;

        }

        # Eğer ki kontrol ediliyor talep var ise ses çalma
        if ($countTransactions['new'] == 0 && $countTransactions['pending'] > 0)
            $isControl = 1;

        # Eğer ki website paneline sahip ise ses çalma
        if ($user->role_id == 3)
            $isControl = 1;

        return response()->json([
            'type' => 'success',
            'message' => 'Başarılı bir şekilde listelendi.',
            'control' => $isControl,
            'list' => $responseData
        ]);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getReports(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id == 4)
            return redirect('/admin/dashboard');

        $getPaymentMethods = PaymentMethod::query()->where('status', 1)->get();
        $getDepositRanges = DepositRange::query()->where('status', 1)->get();
        $getSites = new Site();
        $getSites = $getSites->query();

        if (!empty($user->api_id))
            $getSites = $getSites->where('id', $user->api_id);

        $getSites = $getSites->where('status', 1);
        $getSites = $getSites->get();

        return view('panel.reports')
            ->with('user', $user)
            ->with('sites', $getSites)
            ->with('paymentMethods', $getPaymentMethods)
            ->with('depositRanges', $getDepositRanges);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getReportList(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        $methodId = $request->input('method_id');
        $apiId = $request->input('api_id');
        $playerId = $request->input('player_id');
        $minAmount = $request->input('min_amount');
        $maxAmount = $request->input('max_amount');
        $depositRangeId = $request->input('deposit_range_id');
        $startDate = date('Y-m-d', strtotime('-3 days')) . ' 00:00:00';
        $endDate = date('Y-m-d H:i:s');

        if ($request->method() == 'POST') {
            if (!empty($request->input('filter_start_date')))
                $startDate = $request->input('filter_start_date') . ':00';

            if (!empty($request->input('filter_end_date')))
                $endDate = $request->input('filter_end_date') . ':59';
        }

        # Site Commissions
        $getSiteCommissions = SiteCommission::query()->get();
        $dataSiteCommission = [];

        foreach ($getSiteCommissions as $commission) {
            $dataSiteCommission[$commission->site_id][$commission->payment_method_id] = $commission->percent;
        }

        # Bank Accounts List
        if (!empty($depositRangeId)) {
            $bankAccount = new BankAccount();
            $bankAccount = $bankAccount->query();

            if (!empty($depositRangeId))
                $bankAccount = $bankAccount->where('deposit_range_id', (int)$depositRangeId);

            $bankAccount = $bankAccount->get();

            # Payment Method to Array
            $bankAccounts = [];
            $bankAccounts_in = [];
            foreach ($bankAccount as $plist) {
                $bankAccounts[$plist->id] = $plist;
                $bankAccounts_in[] = $plist->id;
            }
        }

        # Transactions
        $getTransactions = new Transaction();
        $getTransactions = $getTransactions->with('payment_method');
        $getTransactions = $getTransactions->where('updated_at', '>', $startDate);
        $getTransactions = $getTransactions->where('updated_at', '<', $endDate);

        if (!empty($methodId))
            $getTransactions = $getTransactions->where('transaction_type', $methodId);

        if (!empty($playerId))
            $getTransactions = $getTransactions->where('player_id', $playerId);

        if (!empty($apiId))
            $getTransactions = $getTransactions->where('api_id', $apiId);

        if (!empty($depositRangeId)) {
            $getTransactions = $getTransactions->where(function ($query) use ($bankAccounts_in) {
                $query->whereIn('share_bank_account_id', $bankAccounts_in)
                    ->orWhereIn('bank_account_id', $bankAccounts_in);
            });;
        }

        if (!empty($minAmount) && $minAmount > 0)
            $getTransactions = $getTransactions->where('amount', '>=', $minAmount);

        if (!empty($maxAmount) && $maxAmount > 0)
            $getTransactions = $getTransactions->where('amount', '<=', $maxAmount);

        $getTransactions = $getTransactions->orderBy('id', 'DESC');
        $getTransactions = $getTransactions->get();

        $dataTransactions = [];

        foreach ($getTransactions as $transaction) {
            if (empty($dataTransactions[$transaction->api_id][$transaction->payment_method->name]))
                $dataTransactions[$transaction->api_id][$transaction->payment_method->name] = [
                    'paymentMethod' => [
                        'id' => $transaction->payment_method->id,
                        'name' => $transaction->payment_method->name,
                        'type' => $transaction->payment_method->type,
                    ],
                    'api' => [
                        'id' => $transaction->api_id,
                        'name' => $transaction->sites->name
                    ],
                    'counts' => [
                        'deposit' => [
                            'amount' => 0,
                            'piece' => 0,
                            'commission' => 0,
                            'commission_amount' => 0
                        ],
                        'withdraw' => [
                            'amount' => 0,
                            'piece' => 0,
                            'commission' => 0,
                            'commission_amount' => 0
                        ]
                    ]
                ];

            if ($transaction->is_success == 1) {
                if ($transaction->payment_method->type == 1) {
                    $siteCommission = $this->checkCommission(
                        $dataSiteCommission,
                        $transaction->api_id,
                        $transaction->transaction_type
                    );

                    if ($siteCommission > 0) {
                        $commissionAmount = $transaction->amount * $siteCommission / 100;
                    } else {
                        $commissionAmount = 0;
                    }

                    $dataTransactions[$transaction->api_id][$transaction->payment_method->name]['counts']['deposit']['commission'] = $siteCommission;
                    $dataTransactions[$transaction->api_id][$transaction->payment_method->name]['counts']['deposit']['commission_amount'] += $commissionAmount;
                    $dataTransactions[$transaction->api_id][$transaction->payment_method->name]['counts']['deposit']['amount'] += $transaction->amount;
                    $dataTransactions[$transaction->api_id][$transaction->payment_method->name]['counts']['deposit']['piece']++;
                }

                if ($transaction->payment_method->type == 2) {
                    $siteCommission = $this->checkCommission(
                        $dataSiteCommission,
                        $transaction->api_id,
                        $transaction->transaction_type
                    );

                    if ($siteCommission > 0) {
                        $commissionAmount = $transaction->amount * $siteCommission / 100;
                    } else {
                        $commissionAmount = 0;
                    }

                    $dataTransactions[$transaction->api_id][$transaction->payment_method->name]['counts']['withdraw']['commission'] = $siteCommission;
                    $dataTransactions[$transaction->api_id][$transaction->payment_method->name]['counts']['withdraw']['commission_amount'] += $commissionAmount;
                    $dataTransactions[$transaction->api_id][$transaction->payment_method->name]['counts']['withdraw']['amount'] += $transaction->amount;
                    $dataTransactions[$transaction->api_id][$transaction->payment_method->name]['counts']['withdraw']['piece']++;
                }
            }
        }

        return view('panel.report_list')
            ->with('user', $user)
            ->with('transactions', $dataTransactions)
            ->with('startDate', $startDate)
            ->with('endDate', $endDate);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getPlayers(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id == 3 || $user->role_id == 4)
            return redirect('/admin/dashboard');

        $getPaymentMethods = PaymentMethod::where('status', 1)->get();
        $getSites = Site::where('status', 1)->get();

        return view('panel.players')
            ->with('user', $user)
            ->with('sites', $getSites)
            ->with('paymentMethods', $getPaymentMethods);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getPlayerList(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id == 3)
            return redirect('/admin/dashboard');

        $username = $request->input('username');
        $email = $request->input('email');
        $apiId = $request->input('api_id');

        $getPlayers = new Player();

        if (!empty($username))
            $getPlayers = $getPlayers->where('username', 'LIKE', '%'.$username.'%');

        if (!empty($email))
            $getPlayers = $getPlayers->where('email', 'LIKE', '%'.$email.'%');

        if (!empty($apiId))
            $getPlayers = $getPlayers->where('api_id', $apiId);

        $getPlayers = $getPlayers->orderBy('id', 'DESC');
        $getPlayers = $getPlayers->limit(100);
        $getPlayers = $getPlayers->paginate(20);

        return view('panel.player_list')
            ->with('user', $user)
            ->with('players', $getPlayers);
    }

    /**
     * @param $id
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getPlayerShow($id, Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id == 3)
            return redirect('/admin/dashboard');

        $getPlayer = Player::where('id', $id)->first();
        $getPaymentMethods = PaymentMethod::where('status', 1)->get();
        $getSites = Site::where('status', 1)->get();

        if (empty($getPlayer))
            return redirect('/admin/players');

        if ($request->method() == 'POST') {
            if ($user->role_id != 1)
                return back()->with('error', 'Hata, yetkiniz bulunmamaktadır !');

            $email = $request->input('email');
            $firstName = $request->input('first_name');
            $lastName = $request->input('last_name');
            $phoneNumber = $request->input('phone_number');
            $status = $request->input('status');

            $updateData = [
                'email' => $email,
                'first_name' => $firstName,
                'last_name' => $lastName,
                'status' => $status
            ];

            if (!empty($phoneNumber))
                $updateData['phone_number'] = $phoneNumber;

            $updatePlayer = Player::where('id', $id)->update($updateData);

            if ($updatePlayer) {
                Log::addLog($user->id, 4, 2, $id);

                return redirect('/admin/players/show/' . $id)->with('success', 'Oyuncu Düzenlenmiştir !');
            } else {
                return back()->with('error', 'Hata, oyuncu düzenlenmedi !');
            }
        }

        return view('panel.show_player')
            ->with('user', $user)
            ->with('detail', $getPlayer)
            ->with('sites', $getSites)
            ->with('paymentMethods', $getPaymentMethods);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getProfile(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($request->method() == 'POST') {
            $password = $request->input('password');
            $email = $request->input('email');
            $firstName = $request->input('first_name');
            $lastName = $request->input('last_name');

            if (empty($email) || empty($firstName) || empty($lastName))
                return back()->with('error', 'Hata, boş alan bırakmayınız.');

            if ($user->email != $email) {
                $checkEmail = User::where('email', $email)->get();

                if (count($checkEmail) > 0)
                    return back()->with('error', 'Hata, belirtilen email mevcuttur.');
            }

            $newData = [
                'email' => $email,
                'first_name' => $firstName,
                'last_name' => $lastName
            ];

            if (!empty($password))
                $newData['password'] = bcrypt($password);

            $updateData = User::where('id', $user->id)->update($newData);

            if ($updateData) {
                Log::addLog($user->id, 1);

                return redirect('/admin/profile');
            } else {
                return back()->with('error', 'Hata, profil düzenlenemedi !');
            }
        }

        return view('panel.update_profile')
            ->with('user', $user);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getGoogleTwoFactor(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($request->method() == 'POST') {
            $newData = [
                'ga_status' => 1,
                'updated_at' => Carbon::now()
            ];

            $updateData = User::query()->where('id', $user->id)->update($newData);

            if ($updateData) {
                Log::addLog($user->id, 30, 11, $user->id);

                return redirect('/admin/g2a');
            } else {
                return back()->with('error', 'Google Authenticator aktifleştirilirken bir hata oluştu !');
            }
        }

        $ga = new GoogleAuthenticator();
        $secret = $user->ga_secret ?? $ga->createSecret();

        $accountName = env('PAY_WEBSITE_NAME') . ' - ' . $user->username;
        $qrCodeUrl = $ga->getQRCodeGoogleUrl($accountName, $secret);

        # Update G2A Secret for User
        User::query()
            ->where('id', $user->id)
            ->update([
                'ga_secret' => $secret,
                'updated_at' => Carbon::now()
            ]);

        return view('panel.g2a')
            ->with('user', $user)
            ->with('qrCodeUrl', $qrCodeUrl)
            ->with('accountName', $accountName)
            ->with('accountSecret', $secret);
    }

    /**
     * @return RedirectResponse|Redirector
     */
    public function disabledGoogleTwoFactor()
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        # Update G2A for User
        $updateG2a = User::query()
            ->where('id', $user->id)
            ->update([
                'ga_status' => 0,
                'ga_login' => 0,
                'ga_secret' => null,
                'updated_at' => Carbon::now()
            ]);

        if ($updateG2a) {
            Log::addLog($user->id, 29, 11, $user->id);

            return redirect('/admin/g2a');
        } else
            return back()->with('error', 'Google Authenticator kapatılırken bir hata oluştu !');
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function createManualTransaction(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id == 2 && $user->access_manuel_transaction == 0)
            return redirect('/admin/dashboard');

        if ($request->method() == 'POST') {
            $paymentMethod = $request->input('payment_method');
            $type = $request->input('type');
            $username = $request->input('username');
            $firstName = $request->input('first_name');
            $lastName = $request->input('last_name');
            $customerNote = $request->input('customer_note');
            $amount = $request->input('amount');
            $apiId = $request->input('api_id');
            $bankAccountId = $request->input('bank_account_id');
            $accountNumber = $request->input('account_number');
            $iban = $request->input('iban');
            $branchCode = $request->input('branch_code');
            $identityNumber = $request->input('identity_number');
            $identityExpiryDate = $request->input('identity_expiry_date');
            $identityReceiveDate = $request->input('identity_receive_date');
            $processAccountName = $request->input('process_account_name');

            if (
                empty($paymentMethod) ||
                empty($type) ||
                empty($username) ||
                empty($firstName) ||
                empty($lastName) ||
                empty($amount) ||
                empty($apiId)
            )
                return response()->json([
                    'status' => false,
                    'message' => 'Hata, lütfen boş alan bırakmayınız !'
                ]);

            if ($paymentMethod == 1 && empty($bankAccountId))
                return response()->json([
                    'status' => false,
                    'message' => 'Hata, lütfen banka seçiniz !'
                ]);

            # Para Kontrolü
            if (!is_numeric($amount))
                return response()->json([
                    'status' => false,
                    'message' => 'Miktar sayısal bir değer içermelidir !'
                ]);

            # Müşteri Notu Karakter Sayısı
            if (strlen($customerNote) > 255)
                return response()->json([
                    'status' => false,
                    'message' => 'Müşteri notu maksimum 255 karakter olmalıdır !'
                ]);

            $amount = str_replace(',', '', $amount);
            $amount = number_format($amount, 2, '.', '');

            // Select Payment Method Type
            if ($paymentMethod == 1) {
                if ($type == 1) {
                    $paymentMethodType = 1;
                } else {
                    $paymentMethodType = 7;
                }
            } else {
                if ($type == 1) {
                    $paymentMethodType = 3;
                } else {
                    $paymentMethodType = 8;
                }
            }

            if ($user->role_id == 1) {
                $checkPlayer = Player::where('api_id', $apiId)->where('username', $username)->first();

                if (!empty($checkPlayer)) {
                    $playerId = $checkPlayer->id;

                    if (empty($checkPlayer->first_name) && !empty($firstName)) {
                        Player::where('id', $checkPlayer->id)->update(['first_name' => $firstName]);
                    }

                    if (empty($checkPlayer->last_name) && !empty($lastName)) {
                        Player::where('id', $checkPlayer->id)->update(['last_name' => $lastName]);
                    }
                } else {
                    $playerId = Player::insertGetId([
                        'api_id' => $apiId,
                        'username' => $username,
                        'external_user_id' => 0,
                        'external_password' => '',
                        'email' => '',
                        'first_name' => $firstName,
                        'last_name' => $lastName,
                        'phone_number' => '',
                        'status' => 1,
                        'created_at' => Carbon::now(),
                        'updated_at' => Carbon::now()
                    ]);
                }
            } else {
                $checkPlayer = Player::where('api_id', $user->api_id)->where('username', $username)->first();

                if (!empty($checkPlayer)) {
                    $playerId = $checkPlayer->id;

                    if (empty($checkPlayer->first_name) && !empty($firstName)) {
                        Player::where('id', $checkPlayer->id)->update(['first_name' => $firstName]);
                    }

                    if (empty($checkPlayer->last_name) && !empty($lastName)) {
                        Player::where('id', $checkPlayer->id)->update(['last_name' => $lastName]);
                    }
                } else {
                    $playerId = Player::insertGetId([
                        'api_id' => $user->api_id,
                        'username' => $username,
                        'external_user_id' => 0,
                        'external_password' => '',
                        'email' => '',
                        'first_name' => $firstName,
                        'last_name' => $lastName,
                        'phone_number' => '',
                        'status' => 1,
                        'created_at' => Carbon::now(),
                        'updated_at' => Carbon::now()
                    ]);
                }
            }

            $transactionSendData = [
                'process_type' => 'Manuel Olarak Oluşturuldu'
            ];

            foreach ($request->all() as $key => $value) {
                if (!empty($value))
                    $transactionSendData[$key] = $value;
            }

            $privateCode = time();
            $transactionData = [
                'api_id' => $user->role_id == 1 ? $apiId : $user->api_id,
                'transaction_type' => $paymentMethodType,
                'bank_account_id' => 0,
                'bank_id' => $bankAccountId,
                'transaction_send_data' => json_encode($transactionSendData),
                'player_id' => $playerId,
                'amount' => $amount,
                'is_success' => 0,
                'user_id' => 0,
                'log_id' => $privateCode,
                'external_transaction_id' => null,
                'player_approved' => 1,
                'customer_note' => $customerNote,
                'created_at' => Carbon::now(),
                'updated_at' => Carbon::now()
            ];

            $transactionId = Transaction::insertGetId($transactionData);

            if ($transactionId) {
                Log::addLog($user->id, 7, 4, $transactionId);

                return response()->json([
                    'status' => true,
                    'message' => 'Manuel Talep Oluşturma İsteği Oluşturuldu.'
                ]);
            } else {
                return response()->json([
                    'status' => false,
                    'message' => 'Hata, talep oluşturulamadı !'
                ]);
            }
        }

        if (empty($user->api_id))
            $getSites = Site::where('status', 1)->get();
        else
            $getSites = Site::where('id', $user->api_id)->where('status', 1)->get();

        $getBanks = Bank::where('status', 1)->get();
        $getBankAccounts = BankAccount::where('status', 1)->get();

        return view('panel.create_manual_transaction')
            ->with('user', $user)
            ->with('sites', $getSites)
            ->with('bankAccounts', $getBankAccounts)
            ->with('banks', $getBanks);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getDepositRange(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id == 3)
            return redirect('/admin/dashboard');

        $getDepositRange = DepositRange::orderBy('id', 'DESC')->get();

        return view('panel.deposit_range')
            ->with('user', $user)
            ->with('depositRange', $getDepositRange);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function createDepositRange(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        if ($request->method() == 'POST') {
            $name = $request->input('name');
            $min = $request->input('min');
            $max = $request->input('max');
            $status = $request->input('status');

            if (empty($min) || empty($max))
                return back()->with('error', 'Hata, Min ve Max değerler boş olamaz !');

            $insertData = DepositRange::insertGetId([
                'name' => $name,
                'min' => $min,
                'max' => $max,
                'status' => $status
            ]);

            if ($insertData) {
                Log::addLog($user->id, 19, 8, $insertData);

                return redirect('/admin/account-management/deposit-range');
            } else {
                return back()->with('error', 'Hata, kayıt oluşturulamadı !');
            }
        }

        return view('panel.create_deposit_range')
            ->with('user', $user);
    }

    /**
     * @param $id
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function editDepositRange($id, Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id == 3)
            return redirect('/admin/dashboard');

        $getDepositRange = DepositRange::where('id', $id)->first();

        if (empty($getDepositRange))
            return redirect('/admin/account-management/deposit-range');

        if ($request->method() == 'POST') {
            $name = $request->input('name');
            $min = $request->input('min');
            $max = $request->input('max');
            $status = $request->input('status');

            if (empty($min) || empty($max))
                return back()->with('error', 'Hata, Min ve Max değerler boş olamaz !');

            $updateData = DepositRange::where('id', $id)->update([
                'name' => $name,
                'min' => $min,
                'max' => $max,
                'status' => $status
            ]);

            if ($updateData) {
                Log::addLog($user->id, 20, 8, $id);

                return redirect('/admin/account-management/deposit-range');
            } else {
                return back()->with('error', 'Hata, kayıt düzenlenemedi !');
            }
        }

        return view('panel.update_deposit_range')
            ->with('user', $user)
            ->with('detail', $getDepositRange);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getWhiteListIp(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        $getWhiteListIp = WhiteListIp::orderBy('id', 'DESC')->get();

        return view('panel.whitelist_ip')
            ->with('user', $user)
            ->with('whitelist_ip', $getWhiteListIp);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function createWhiteListIp(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        if ($request->method() == 'POST') {
            $ipAddress = $request->input('ip_address');
            $description = $request->input('description');
            $status = $request->input('status');

            if (empty($ipAddress))
                return back()->with('error', 'Hata, lütfen boş alan bırakmayınız !');

            $checkWhiteListIp = WhiteListIp::where('ip_address', $ipAddress)->count();

            if ($checkWhiteListIp > 0)
                return back()->with('error', 'Hata, belirtilen ip adresi mevcuttur.');

            $insertData = WhiteListIp::insertGetId([
                'ip_address' => $ipAddress,
                'description' => $description,
                'status' => $status
            ]);

            if ($insertData) {
                Log::addLog($user->id, 11, 5, $insertData);

                return redirect('/admin/whitelist-ip');
            } else {
                return back()->with('error', 'Hata, kayıt oluşturulamadı !');
            }
        }

        return view('panel.create_whitelist_ip')
            ->with('user', $user);
    }

    /**
     * @param $id
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function updateWhiteListIp($id, Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        $getWhiteListIp = WhiteListIp::where('id', $id)->first();

        if (empty($getWhiteListIp))
            return redirect('/admin/whitelist-ip');

        if ($request->method() == 'POST') {
            $ipAddress = $request->input('ip_address');
            $description = $request->input('description');
            $status = $request->input('status');

            $updateData = WhiteListIp::where('id', $id)->update([
                'ip_address' => $ipAddress,
                'description' => $description,
                'status' => $status
            ]);

            if ($updateData) {
                Log::addLog($user->id, 12, 5, $id);

                return redirect('/admin/whitelist-ip');
            } else {
                return back()->with('error', 'Hata, kayıt düzenlenemedi !');
            }
        }

        return view('panel.update_whitelist_ip')
            ->with('user', $user)
            ->with('detail', $getWhiteListIp);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getPaymentMethods(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        $getPaymentMethods = PaymentMethod::orderBy('id', 'DESC')->get();

        return view('panel.payment_methods')
            ->with('user', $user)
            ->with('payment_methods', $getPaymentMethods);
    }

    /**
     * @param $id
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function updatePaymentMethod($id, Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        $getPaymentMethod = PaymentMethod::where('id', $id)->first();

        if (empty($getPaymentMethod))
            return redirect('/admin/payment-methods');

        if ($request->method() == 'POST') {
            $type = $request->input('type');
            $name = $request->input('name');
            $logo = $request->input('logo');
            $minAmount = $request->input('min_amount');
            $maxAmount = $request->input('max_amount');
            $fee = $request->input('fee');
            $time = $request->input('time');
            $cashBonus = $request->input('cash_bonus');
            $sort = $request->input('sort');
            $status = $request->input('status');

            if (strlen($sort) > 3)
                return back()->with('error', 'Hata, sıra alanı maksimum 2 karakter olmalıdır !');

            $updateData = PaymentMethod::where('id', $id)->update([
                'type' => $type,
                'name' => $name,
                'logo' => $logo,
                'min_amount' => $minAmount,
                'max_amount' => $maxAmount,
                'fee' => $fee,
                'time' => $time,
                'cash_bonus' => $cashBonus,
                'sort' => $sort,
                'status' => $status
            ]);

            if ($updateData) {
                Log::addLog($user->id, 3, 3, $id);

                return redirect('/admin/payment-methods');
            } else {
                return back()->with('error', 'Hata, kayıt düzenlenemedi !');
            }
        }

        return view('panel.update_payment_method')
            ->with('user', $user)
            ->with('detail', $getPaymentMethod);
    }

    /**
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getSettings()
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        $dataEnvironment = json_encode([
            'name' => env('PAY_WEBSITE_NAME'),
            'logo' => env('PAY_WEBSITE_LOGO'),
            'icon' => env('PAY_WEBSITE_ICON'),
            'bg1' => env('PAY_WEBSITE_BACKGROUND1'),
            'bg2' => env('PAY_WEBSITE_BACKGROUND2'),
            'bg3' => env('PAY_WEBSITE_BACKGROUND3'),
            'url' => env('PAY_WEBSITE_URL'),
            'bank_transfer_type' => env('BANK_TRANSFER_TYPE'),
            'papara_type' => env('PAPARA_TYPE'),
            'demo_api' => env('DEMO_API'),
            'system_status' => env('SYSTEM_STATUS')
        ]);

        $getSites = Site::query()->where('status', 1)->get();

        return view('panel.settings')
            ->with('user', $user)
            ->with('env', $dataEnvironment)
            ->with('sites', $getSites);
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function updateSettings(Request $request)
    {
        if (!auth()->check())
            return response()->json([
                'status' => false,
                'message' => 'Hata, üye girişi bulunmamaktadır !'
            ]);

        $user = auth()->user();

        if ($user->role_id != 1)
            return response()->json([
                'status' => false,
                'message' => 'Hata, bu alanı düzenlemek için yetkiniz bulunmamaktadır !'
            ]);

        $checkPermission = substr(sprintf('%o', fileperms(base_path('.env'))), -4);

        if ($checkPermission != '0777')
            return response()->json([
                'status' => false,
                'message' => 'Hata, dosya izni ' . $checkPermission . ' hatalıdır !'
            ]);

        $name = $request->input('name');
        $logo = $request->input('logo');
        $favicon = $request->input('favicon');
        $url = $request->input('url');
        $bg1 = $request->input('bg1');
        $bg2 = $request->input('bg2');
        $bg3 = $request->input('bg3');
        $bankTransferType = $request->input('bank_transfer_type');
        $paParaType = $request->input('papara_type');
        $demoSite = $request->input('demo_site');
        $systemStatus = $request->input('system_status');

        if (
            empty($name) ||
            empty($logo) ||
            empty($favicon) ||
            empty($url) ||
            empty($bankTransferType) ||
            empty($paParaType)
        )
            return response()->json([
                'status' => false,
                'message' => 'Hata, boş alan bırakmayınız !'
            ]);

        $path = base_path('.env');

        if (file_exists($path)) {
            file_put_contents($path, str_replace(
                'PAY_WEBSITE_NAME=' . env('PAY_WEBSITE_NAME'),
                'PAY_WEBSITE_NAME=' . $name,
                file_get_contents($path)
            ));
            file_put_contents($path, str_replace(
                'PAY_WEBSITE_LOGO=' . env('PAY_WEBSITE_LOGO'),
                'PAY_WEBSITE_LOGO=' . $logo,
                file_get_contents($path)
            ));
            file_put_contents($path, str_replace(
                'PAY_WEBSITE_ICON=' . env('PAY_WEBSITE_ICON'),
                'PAY_WEBSITE_ICON=' . $favicon,
                file_get_contents($path)
            ));
            file_put_contents($path, str_replace(
                'PAY_WEBSITE_BACKGROUND1=' . env('PAY_WEBSITE_BACKGROUND1'),
                'PAY_WEBSITE_BACKGROUND1=' . $bg1,
                file_get_contents($path)
            ));
            file_put_contents($path, str_replace(
                'PAY_WEBSITE_BACKGROUND2=' . env('PAY_WEBSITE_BACKGROUND2'),
                'PAY_WEBSITE_BACKGROUND2=' . $bg2,
                file_get_contents($path)
            ));
            file_put_contents($path, str_replace(
                'PAY_WEBSITE_BACKGROUND3=' . env('PAY_WEBSITE_BACKGROUND3'),
                'PAY_WEBSITE_BACKGROUND3=' . $bg3,
                file_get_contents($path)
            ));
            file_put_contents($path, str_replace(
                'PAY_WEBSITE_URL=' . env('PAY_WEBSITE_URL'),
                'PAY_WEBSITE_URL=' . $url,
                file_get_contents($path)
            ));
            file_put_contents($path, str_replace(
                'PAPARA_TYPE=' . env('PAPARA_TYPE'),
                'PAPARA_TYPE=' . $paParaType,
                file_get_contents($path)
            ));
            file_put_contents($path, str_replace(
                'BANK_TRANSFER_TYPE=' . env('BANK_TRANSFER_TYPE'),
                'BANK_TRANSFER_TYPE=' . $bankTransferType,
                file_get_contents($path)
            ));
            file_put_contents($path, str_replace(
                'DEMO_API=' . env('DEMO_API'),
                'DEMO_API=' . $demoSite,
                file_get_contents($path)
            ));
            file_put_contents($path, str_replace(
                'SYSTEM_STATUS=' . env('SYSTEM_STATUS'),
                'SYSTEM_STATUS=' . $systemStatus,
                file_get_contents($path)
            ));

            Log::addLog($user->id, 2);
        } else {
            return response()->json([
                'status' => false,
                'message' => 'Hata, ayarlar dosyası bulunamamıştır !'
            ]);
        }

        return response()->json([
            'status' => true,
            'message' => 'Genel Ayarlar Başarıyla Düzenlenmiştir !'
        ]);
    }

    /**
     * @param Request $request
     * @return JsonResponse
     */
    public function updateManualTransaction(Request $request)
    {
    /*    return response()->json([
            'status' => false,
            'message' => 'HATA, yarrağım :D !'
        ]);*/


        if (!auth()->check())
            return response()->json([
                'status' => false,
                'message' => 'Hata, üye girişi bulunmamaktadır !'
            ]);

        $user = auth()->user();

        if ($user->role_id != 1)
            return response()->json([
                'status' => false,
                'message' => 'Hata, bu alanı düzenlemek için yetkiniz bulunmamaktadır !'
            ]);

        $amount = $request->input('amount');
        $status = $request->input('status');
        $createdAt = $request->input('created_at');
        $id = $request->input('id');
        $payNote = $request->input('pay_note');
        $transactionType = $request->input('transaction_type');

        if (
            empty($id) ||
            empty($createdAt) ||
            empty($transactionType) ||
            empty($amount)
        )
            return response()->json([
                'status' => false,
                'message' => 'Hata, boş alan bırakmayınız !'
            ]);

        $updateTransaction = Transaction::where('id', $id)->update([
            'transaction_type' => $transactionType,
            'amount' => $amount,
            'is_success' => $status,
            'created_at' => $createdAt,
            'pay_note' => $payNote
        ]);

        if (!$updateTransaction)
            return response()->json([
                'status' => false,
                'message' => 'Hata, işlem düzenlenirken bir sorun oluştu !'
            ]);

        Log::addLog($user->id, 8, 4, $id);

        return response()->json([
            'status' => true,
            'message' => 'İşlem Manuel Olarak Güncellendi'
        ]);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getLogs(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        return view('panel.logs')
            ->with('user', $user);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getLogList(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        $typeId = $request->input('type_id');
        $processId = $request->input('process_id');
        #$startDate = date('Y-m-d', strtotime('-3 days')) . ' 00:00:00';
        #$endDate = date('Y-m-d H:i:s');

        if ($request->method() == 'POST') {
            if (!empty($request->input('filter_start_date')))
                $startDate = $request->input('filter_start_date') . ':00';

            if (!empty($request->input('filter_end_date')))
                $endDate = $request->input('filter_end_date') . ':59';
        }

        $getLogs = new Log();

        if ($user->role_id != 1)
            $getLogs = $getLogs->where('user_id', $user->id);

        if (!empty($typeId))
            $getLogs = $getLogs->where('type_id', $typeId);

        if (!empty($processId))
            $getLogs = $getLogs->where('process_id', $processId);

        if (!empty($startDate))
            $getLogs = $getLogs->where('created_at', '>', $startDate);

        if (!empty($endDate))
            $getLogs = $getLogs->where('created_at', '<', $endDate);

        $getLogs = $getLogs->orderBy('id', 'DESC');
        $getLogs = $getLogs->paginate(20);

        return view('panel.log_list')
            ->with('user', $user)
            ->with('logs', $getLogs);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getStatisticOperators(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        #if ($user->role_id != 1)
            #return redirect('/admin/dashboard');

        return view('panel.statistic_operators')
            ->with('user', $user);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getApiLogs(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        return view('panel.api_logs')
            ->with('user', $user);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getApiLogList(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        $typeId = $request->input('type_id');
        $processId = $request->input('process_id');

        if ($request->method() == 'POST') {
            if (!empty($request->input('filter_start_date')))
                $startDate = $request->input('filter_start_date') . ':00';

            if (!empty($request->input('filter_end_date')))
                $endDate = $request->input('filter_end_date') . ':59';
        }

        $getApiLogs = new RequestLog();
        $getApiLogs = $getApiLogs->query();

        if (!empty($processId))
            $getApiLogs = $getApiLogs->where('transaction_id', $processId);

        if (!empty($typeId))
            $getApiLogs = $getApiLogs->where('type_id', $typeId);

        if (!empty($startDate))
            $getApiLogs = $getApiLogs->where('created_at', '>', $startDate);

        if (!empty($endDate))
            $getApiLogs = $getApiLogs->where('created_at', '<', $endDate);

        $getApiLogs = $getApiLogs->orderBy('id', 'DESC');
        $getApiLogs = $getApiLogs->paginate(20);

        return view('panel.api_log_list')
            ->with('user', $user)
            ->with('apiLogs', $getApiLogs);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getCronJobs(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        return view('panel.cron_jobs')
            ->with('user', $user);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getCronJobList(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        $typeId = $request->input('type_id');
        $processId = $request->input('process_id');

        if ($request->method() == 'POST') {
            if (!empty($request->input('filter_start_date')))
                $startDate = $request->input('filter_start_date') . ':00';

            if (!empty($request->input('filter_end_date')))
                $endDate = $request->input('filter_end_date') . ':59';
        }

        $getCronJobs = new CronJob();
        $getCronJobs = $getCronJobs->query();

        if (!empty($processId))
            $getCronJobs = $getCronJobs->where('id', $processId);

        if (!empty($typeId))
            $getCronJobs = $getCronJobs->where('type_id', $typeId);

        if (!empty($startDate))
            $getCronJobs = $getCronJobs->where('created_at', '>', $startDate);

        if (!empty($endDate))
            $getCronJobs = $getCronJobs->where('created_at', '<', $endDate);

        $getCronJobs = $getCronJobs->orderBy('id', 'DESC');
        $getCronJobs = $getCronJobs->paginate(20);

        return view('panel.cron_job_list')
            ->with('user', $user)
            ->with('cronJobs', $getCronJobs);
    }

    /**
     * @param $id
     * @return Application|JsonResponse|RedirectResponse|Redirector
     */
    public function runCronJob($id)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        if ($user->role_id != 1)
            return redirect('/admin/dashboard');

        # Empty Delete Transactions
        if ($id) {
            $url = env('APP_URL') . '/cron/hourly/delete-empty-transactions?userId=' . $user->id;
            $ch = curl_init();
            curl_setopt($ch, CURLOPT_URL, $url);
            curl_exec($ch);
            curl_close($ch);

            return response()->json([
                'type' => 'success',
                'message' => 'success'
            ]);
        }

        return response()->json([
            'type' => 'error',
            'message' => 'Bilinmeyen Cron Job İşlemi'
        ]);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getPushBullet(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        return view('panel.push_bullet')
            ->with('user', $user);
    }

    /**
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getPushBulletList(Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();

        $status = $request->input('status');

        $getSms = new PushBulletSms();

        if ($status != '')
            $getSms = $getSms->where('status', $status);

        $getSms = $getSms->orderBy('id', 'DESC');
        $getSms = $getSms->paginate(20);

        return view('panel.push_bullet_list')
            ->with('user', $user)
            ->with('sms', $getSms);
    }

    /**
     * @param $id
     * @param Request $request
     * @return Factory|RedirectResponse|Redirector|View
     */
    public function getPushBulletDetail($id, Request $request)
    {
        if (!auth()->check())
            return redirect('/admin/login');

        $user = auth()->user();
        $getSms = PushBulletSms::where('id', $id)->first();
        $getSmsSeen = PushBulletSeen::where('sms_id', $id)->get();

        if (!empty($getSms)) {
            PushBulletSms::where('id', $id)
                ->where('status', 0)
                ->update([
                    'status' => 1,
                    'updated_at' => Carbon::now()
                ]);

            # Insert Seen
            $countSmsSeen = PushBulletSeen::where('sms_id', $id)->where('user_id', $user->id)->count();

            if ($countSmsSeen == 0) {
                PushBulletSeen::insert([
                    'sms_id' => $id,
                    'user_id' => $user->id,
                    'created_at' => Carbon::now(),
                    'updated_at' => Carbon::now()
                ]);
            }
        }

        return view('panel.push_bullet_detail')
            ->with('user', $user)
            ->with('sms', $getSms)
            ->with('detailId', $id)
            ->with('seenList', $getSmsSeen);
    }

    /**
     * @param $id
     * @return JsonResponse
     */
    public function getPushBulletDelete($id)
    {
        if (!auth()->check())
            return response()->json([
                'type' => 'error',
                'message' => 'Lütfen, üye girişi yapınız !'
            ]);

        $user = auth()->user();

        if ($user->access_push_bullet == 0)
            return response()->json([
                'type' => 'error',
                'message' => 'Hata, yetkiniz bulunmamaktadır !'
            ]);

        $getAccount = PushBulletSms::where('id', $id)->first();

        if (empty($getAccount))
            return response()->json([
                'type' => 'error',
                'message' => 'Hata, kayıt bulunamadı !'
            ]);

        $updateData = PushBulletSms::where('id', $id)->update([
            'status' => 2,
        ]);

        if ($updateData) {
            Log::addLog($user->id, 26, 10, $id);

            return response()->json([
                'type' => 'success',
                'message' => 'Mesaj Silinmiştir.'
            ]);
        } else {
            return response()->json([
                'type' => 'error',
                'message' => 'Hata, kayıt silinemedi !'
            ]);
        }
    }

    /**
     * @param $apiId
     * @param $apiKey
     * @param $userId
     * @param $username
     * @param $amount
     * @param $type
     * @param $externalTransactionId
     * @return string
     */
    private function hashCalculate(
        $apiId,
        $apiKey,
        $userId,
        $username,
        $amount,
        $type,
        $externalTransactionId
    )
    {
        $hashParameters = [
            'api_id' => $apiId,
            'api_key' => $apiKey,
            'user_id' => $userId,
            'username' => $username,
            'amount' => $amount,
            'type' => $type
        ];

        if (!empty($externalTransactionId))
            $hashParameters['external_transaction_id'] = $externalTransactionId;

        $generateHash = hash("sha512", http_build_query($hashParameters));

        return $generateHash;
    }

    /**
     * @param $responseHash
     * @param $generateHash
     * @return bool
     */
    private function checkHash($responseHash, $generateHash)
    {
        if (!empty($responseHash) && $responseHash == $generateHash)
            return true;
        else
            return false;
    }

    /**
     * @param $data
     * @param $siteId
     * @param $paymentMethodId
     * @return float|int
     */
    private function checkCommission($data, $siteId, $paymentMethodId)
    {
        if (!empty($data[$siteId][$paymentMethodId]))
            return round($data[$siteId][$paymentMethodId] / (100 / 100),2);
        else
            return 0;
    }
}
